<?php
// *****************************************************************************
// Copyright 2003-2005 by A J Marston <http://www.tonymarston.net>
// Copyright 2006-2024 by Radicore Software Limited <http://www.radicore.org>
// *****************************************************************************
// $Date: 2024-07-30 11:31:16 +0100 (Tue, 30 Jul 2024) $
// $Author: tony $
// $Revision: 1574 $
// *****************************************************************************

#[\AllowDynamicProperties]
class sqlsrv
// this version is for SQL Server
{
    // connection settings
    var $dbname;                        // database name
    var $dbname_prev='???';             // previous database name
    var $dbschema;                      // schema name
    var $dbprefix;
    var $serverName;
    var $connectionInfo;

    // server settings
    var $client_info = '';              // output from sqlsrv_client_info()
    var $server_info = '';              // output from sqlsrv_server_info()

    var $alias_names = array();         // array of 'alias=expression' clauses
    var $audit_logging;                 // yes/no switch
    var $errors;                        // array of errors
    var $error_string;                  //
    var $fieldspec = array();           // field specifications (see class constructor)
    var $lastpage;                      // last available page number in current query

    // by default if the database cannot acquire a lock it will abort, but this behaviour can be changed
    var $lock_wait_count=0;             // number of times lock wait failed
    var $no_abort_on_lock_wait=false;   // do not abort if cannot apply lock, allow me to try again
    var $no_read_lock=false;            // do not lock this table when reading from it

    var $no_duplicate_error;            // if TRUE do not create an error when inserting a duplicate
    var $numrows;                       // number of rows retrieved
    var $pageno;                        // requested page number
    var $primary_key = array();         // array of primary key names
    var $retry_on_duplicate_key;        // field name to be incremented when insert fails
    var $rows_per_page;                 // page size for multi-row forms
    var $row_locks;                     // SH=shared, EX=exclusive
    var $row_locks_supp;                // supplemental lock type
    var $skip_offset;                   // force diferent offset after rows have been skipped
    //var $table_locks;                   // array of tables to be locked
    var $temp_tables;                   // array of temporary table names being used
    var $temporary_table;               // name of table to be used in this query
    var $transaction_level;             // transaction level
    var $unique_keys = array();         // array of candidate keys
    var $update_on_duplicate_key;       // switch to 'update' if insert fails

    // the following are used to construct an SQL query
    var $sql_select;
    var $sql_from;
    var $sql_groupby;
    var $sql_having;
    var $sql_orderby;
    var $sql_orderby_seq;               // 'asc' or 'desc'
    var $sql_where_append;              // string which is too complex for where2array() function
    var $sql_union;
    var $query;                         // completed DML statement

    // these are used in Common Table Expressions (CTE)
    var $sql_CTE_name;                  // CTE name
    var $sql_CTE_select;                // CTE 'select columns'
    var $sql_CTE_anchor;                // CTE anchor expression
    var $sql_CTE_recursive;             // CTE recursive expression

    // these are used to construct a subquery nested within the FROM clause
    var $sql_derived_table;             //
    var $sql_derived_select;            //
    var $sql_derived_from;              //
    var $sql_derived_where;             //
    var $sql_derived_search;            //
    var $sql_derived_groupby;           //
    var $sql_derived_having;            //
    var $sql_derived_orderby;           //

    var $dbconnect;                     // database connection resource

    // ****************************************************************************
    // class constructor
    // ****************************************************************************
    function __construct ($args=null)
    {
        if (is_string($args)) {
            $dbname               = $args;
        } else {
            $this->dbschema       =& $args['SQLSRV_schema'];
            $dbname               =& $args['dbname'];
            $this->dbprefix       =& $args['dbprefix'];
            $this->serverName     =& $args['serverName'];
            $this->connectionInfo =& $args['connectionInfo'];
        } // if

        if (!empty($dbname)) {
            $dmlobject = $this;  // ensure this is in $errcontext if an error occurs
            $result = $this->connect($dbname) or trigger_error('SQLSRV', E_USER_ERROR);
        } else {
            $result = TRUE;
        } // if

        if (defined('TRANSIX_NO_AUDIT') OR defined('NO_AUDIT_LOGGING')) {
        	// do nothing
        } else {
            if (!class_exists('audit_tbl')) {
        	    // obtain definition of the audit_tbl class
        		require_once 'classes/audit_tbl.class.inc';
        	} // if
        } // if

        $this->temp_tables = array();

        return $result;

    } // __construct

    // ****************************************************************************
    function __destruct ()
    {
        if (is_resource($this->dbconnect)) {
            $res = sqlsrv_close ($this->dbconnect);
        } // if

    } // __destruct

    // ****************************************************************************
    function adjustData ($string_in)
    // modify string to escape any single quote with a second single quote
    // (do not use backslash as with MySQL)
    {
        if (is_null($string_in)) {
            $string_in = '';  // set to empty string (for version 8.1)
        } // if

        $pattern = <<< END_OF_PATTERN
/
^                       # begins with
\w+[ ]*\(               # 'function('
(                       # start choice
\w+                     # word
|                       # or
\(.+?\)                 # '(...)' string inside parentheses
|,[ ]*                  # or ','
'(?P<quoted_string>.+)' # '<quoted_string>' named pattern
)+                      # end choice
/imsx
END_OF_PATTERN;

        $string_out = '';
        if (preg_match($pattern, $string_in, $regs, PREG_OFFSET_CAPTURE)) {
            // found "function(......)" so see if it contains a 'quoted_string'
            if (isset($regs['quoted_string'])) {
                $regs['quoted_string'][0] = trim($regs['quoted_string'][0]);
                if (!empty($regs['quoted_string'][0])) {
                    // a quoted string has been found, so escape any quotes within it
                    $quoted_string = $regs['quoted_string'][0];
                    $length1 = strlen($quoted_string);
                    $offset1 = $regs['quoted_string'][1];
                    $quoted_string = str_replace("'", "''", $quoted_string);
                    $string_out = substr_replace($string_in, $quoted_string, $offset1, $length1);
                } else {
                    $string_out = str_replace("'", "''", $string_in);
                } // if
            } else {
                $string_out = str_replace("'", "''", $string_in);
            } // if
        } else {
            $string_out = str_replace("'", "''", $string_in);
        } // if

        if (preg_match('/(?<now>\bnow\(\))/i', $string_out, $regs)) {
            $string_out = preg_replace('/\bnow\(\)/i', "SYSDATETIME()", $string_out);
        } // if

        return $string_out;

    } // adjustData

    // ****************************************************************************
    function adjust_derived_query ($select_str, $from_str, $having_str, $search_str, $tablename)
    // insert a nested subquery for a derived table into the FROM clause
    {
        $subquery   = null;
        $append_str = null;

        $having_str = mergeWhere($having_str, $search_str);

        // make any adjustments to the component parts
        $sql_derived_select = $this->adjustSelect($this->sql_derived_select);

        if (!empty($this->sql_derived_where)) {
            $sql_derived_where = $this->adjustWhere($this->sql_derived_where);
        } else {
            $sql_derived_where = null;
        } // if

        $sql_derived_from = $this->adjustFrom($this->sql_derived_from, $sql_derived_where);

        if (!empty($this->sql_derived_groupby)) {
            $sql_derived_groupby = $this->adjustGroupBy($sql_derived_select, $this->sql_derived_groupby, $this->sql_derived_orderby);
        } else {
            $sql_derived_groupby = null;
        } // if

        if (!empty($this->sql_derived_orderby)) {
            $sql_derived_orderby = $this->adjustOrderBy($this->sql_derived_orderby, $sql_derived_select, $sql_derived_from, $sql_derived_groupby, $this->sql_derived_table);
        } else {
            $sql_derived_orderby = null;
        } // if

        //if (!empty($this->sql_derived_having)) {
        //    $sql_derived_having = $this->adjustHaving ($sql_derived_select, $sql_derived_from, $sql_derived_where, $sql_derived_groupby, $sql_derived_having, $sql_derived_orderby);
        //} else {
        //    $sql_derived_having = null;
        //} // if

        // construct the subquery
        if (!empty($having_str)) {
            // include this subquery in another subquery to combine the HAVING clause
            $subquery  = "(\nSELECT $select_str \nFROM ";
            $subquery .= "(\nSELECT $sql_derived_select";
            $subquery .= "\nFROM $sql_derived_from";
            if (!empty($sql_derived_where)) {
                $subquery .= "\nWHERE $sql_derived_where";
            } // if
            if (!empty($sql_derived_groupby)) {
                $subquery .= "\nGROUP BY $sql_derived_groupby";
            } // if
            if (!empty($sql_derived_orderby)) {
                $subquery .= "\nORDER BY $sql_derived_orderby";
            } // if
            $subquery .= "\n) AS $this->sql_derived_table";

            // replace the original $select_str with a new one so that the
            // HAVING string can be moved to the outer query
            $select_str = "*";
            //$append_str .= "\n) AS y";
            $append_str .= "\n) AS $tablename";  // set alias to original tablename for $having_str
            $append_str .= "\nWHERE $having_str";

        } else {
            // build standard subquery
            $subquery  = "(\nSELECT $sql_derived_select";
            $subquery .= "\nFROM $sql_derived_from";
            if (!empty($sql_derived_where)) {
                $subquery .= "\nWHERE $sql_derived_where";
            } // if
            if (!empty($sql_derived_groupby)) {
                $subquery .= "\nGROUP BY $sql_derived_groupby";
            } // if
            //if (!empty($sql_derived_having)) {
            //    $subquery .= "\nHAVING $sql_derived_having";
            //} // if
            if (!empty($sql_derived_orderby)) {
                $subquery .= "\nORDER BY $sql_derived_orderby";
            } // if
            $subquery .= "\n) AS $this->sql_derived_table";
        } // if

        // replace the reference to this subquery by the subquery itself
        $length   = strlen((string)$this->sql_derived_table);
        $from_str = substr_replace($from_str, $subquery, 0, $length);

        return array($select_str, $from_str, $append_str);

    } // adjust_derived_query

    // ****************************************************************************
    function adjustCAST ($input)
    // replace "CAST('N' AS TYPE(1)) AS alias" with correct TYPE for SQL Server
    {
        $output = $input;

        $pattern1 = <<< END_OF_REGEX
/
CAST\s*                           # CAST
\([ ]*                            # (
(?<string>'(([^\\\']*(\\\.)?)*)') # quoted string
\s+AS\s+(?<type>\w+)              # AS type
(\((?<size>[a-z0-9]+(,\d+)?)\))?  # optional (size,scale)
\)                                # )
\s+AS\s+(?<alias>\w+)             # AS alias
/xims
END_OF_REGEX;

        if ($count = preg_match_all($pattern1, $input, $regs)) {
            foreach ($regs[0] as $string1) {
                $array = array();
                $count = preg_match($pattern1, $string1, $regs2);
                $old_string = $regs2[0];
                $regs2['type'] = strtoupper($regs2['type']);
                switch ($regs2['type']) {
                    case 'CHAR':
                        $type = 'NVARCHAR';
                        if (!empty($regs2['size'])) {
                            $type .= "(".$regs2['size'].")";
                        } // if
                        break;
                    case 'SIGNED':
                        $type = 'INTEGER';
                        break;
                    case 'UNSIGNED':
                        $type = 'INTEGER';
                        break;
                    case 'DECIMAL':
                        $type = 'DECIMAL';
                        if (!empty($regs2['size'])) {
                            $type .= "(".$regs2['size'].")";
                        } // if
                        break;
                    default:
                        $type = $regs2['type'];
                }  // switch
                $new_string = "CAST({$regs2['string']} AS {$type}) AS [{$regs2['alias']}]";
                $single = 1;
                $output  = str_replace($old_string, $new_string, $output, $single);
            } // foreach;
        } // if

        return $output;

    } // adjustCAST

    // ****************************************************************************
    function adjustCONCAT ($input)
    // replace 'CONCAT(A, B, C)' with 'A + B + C'.
    {
        $output = $input;

        $pattern1 = <<< END_OF_REGEX
/
(?<=\bconcat\b\()            # 'concat('
(                            # start choice
 \w+[ ]*\([^\(\)]*\)         # 'FUNC(...)'
 |
 '(([^\\\']*(\\\.)?)*)'      # quoted string
 |
 \w+(\.\w+)?                 # 'word' or 'word.word'
 |
 ,                           # comma
 |
 [ ]*                        # 0 or more spaces
)                            # end choice
*                            # 0 or more times
[ ]*                         # 0 or more spaces
/xims
END_OF_REGEX;

        $pattern2 = <<< END_OF_REGEX
/
\w+[ ]*\([^\(\)]*\)         # 'FUNC(...)'
|
'(([^\\\']*(\\\.)?)*)'      # quoted string
|
\w+\.\w+                    # word dot word
|
\w+                         # word
/xims
END_OF_REGEX;

        if ($count = preg_match_all($pattern1, $input, $regs)) {
            foreach ($regs[0] as $string1) {
                $array = array();
                $count = preg_match_all($pattern2, $string1, $regs2);
                foreach ($regs2[0] as $value) {
                    // trim leading and trailing spaces from each entry
                    if (substr($value, 0, 1) == "'") {
                        // value is quoted so use as-is
                        $array[] = trim($value);
                    } else {
                        // this is a field, so cast it to string to avoid conversion errors
                        $array[] = "CAST(".trim($value)." AS NVARCHAR(MAX))";
                    } // if
                } // foreach
                $string2 = implode('+', $array);     // rejoin with '+' as separator
                // escape any '(' and ')' for use in a pattern
                $string1 = str_replace('(', "\(", $string1);
                $string1 = str_replace(')', "\)", $string1);
                $string1 = str_replace('[', "\[", $string1);
                $string1 = str_replace(']', "\]", $string1);
                $string1 = str_replace('.', "\.", $string1);
                $string1 = str_replace('?', "\?", $string1);
                $string1 = str_replace('+', "\+", $string1);
                $pattern = '#concat\(' .$string1 .'\)#iu';
                $output  = preg_replace($pattern, $string2, $output);
            } // foreach;
        } // if

        return $output;

    } // adjustCONCAT

    // ****************************************************************************
    function adjustFrom ($from_str, $where_str)
    // make any adjustments to the FROM string
    {
        if (!empty($this->temp_tables)) {
            $from_str = trim($from_str);
            if (preg_match('/^(\w+)$/', $from_str)) {
                // $from_str contains a single word, but is this the name of a temporary table?
                if (array_key_exists($from_str, $this->temp_tables)) {
                    $from_str = '#'.$from_str;  // yes, so prefix with '#'
                } // if
                return $from_str;
            } // if
        } // if

        // look for 'CROSS JOIN x ON (...)' and move 'ON (...)' to end of $where string
        if (preg_match('/CROSS[ ]JOIN[ ]+\w*[ ]+(?<on>ON[ ]+.*?)(?=(LEFT|RIGHT))/ims', $from_str, $regs)) {
            $on_str = $regs['on'];
            // remove ' ON (...)' from $from_str
            $from_str = str_ireplace($on_str, '', $from_str);
            // change ' ON ' to ' AND ' and append to $where_str
            $on_str = str_ireplace('ON ', ' AND ', $on_str);
            $where_str .= $on_str;
        } // if

        if (preg_match('/[ ]+AS[ ]+unsigned[ ]+int/i', $from_str)) {
            $from_str = preg_replace('/[ ]+AS[ ]+unsigned[ ]+int/iu', ' AS int', $from_str);
        } // if

        $from_str = $this->adjustFirstValue($from_str);

        //$from_str = $this->adjustCONCAT($from_str);

        // search $output and replace any '=alias' with '=expression'
        $test_str = $this->replaceAliasWithExpression($from_str, $this->alias_names, '=');
        if (!empty($test_str)) {
            $from_str = $test_str;
        } // if

        return $from_str;

    } // adjustFrom

    // ****************************************************************************
    function adjustFirstValue ($input)
    // look for any calls to FIRST_VALUE or LAST_VALUE and change
    // from: [DISTINCT] FIRST_VALUE(col) OVER (ORDER BY col) FROM ... WHERE ... [LIMIT 1].
    //   to: [DISTINCT] FIRST_VALUE(col) OVER (ORDER BY col) FROM ... WHERE ...
    {
        $output = $input;

        $pattern = <<<END_OF_REGEX
/
(?<distinct>DISTINCT[ ])?         # 'DISTINCT ' (optional)
(?<function>                      # function name
\b(FIRST_VALUE|LAST_VALUE)\b
)
[ ]*\([ ]*                        # ' ( '
(?<value>(\w+\.)?\w+){1}          # [word.]word
[ ]*\)[ ]*OVER[ ]*\(              # ') OVER ('
[ ]*ORDER[ ]BY[ ]*                # 'ORDER BY'
(?<orderby>(
(\w+\.)?\w+(\s(ASC|DESC))?
))+
\)\s*FROM[ ]+                     # ') FROM '
(?<from>.+?)                      # FROM string
\s*WHERE[ ]{1}                    # ' WHERE '
(?<where>.+?)                     # WHERE string
(                                 # start choice
(?<limit>LIMIT\s+\d)              # 'LIMIT n' (optional)
|                                 # or
(?<terminator>\))                 # ')'
)                                 # end choice
/imsx
END_OF_REGEX;

        $offset = 0;
        while ($count = preg_match($pattern, $output, $regs, PREG_OFFSET_CAPTURE, $offset)) {
            $string1   = $regs[0][0];
            $str_start = $regs[0][1];

            $function  = $regs['function'][0];
            $value     = $regs['value'][0];
            $orderby   = $regs['orderby'][0];
            $from      = $regs['from'][0];
            $where     = $regs['where'][0];
            $distinct   =& $regs['distinct'][0];    // optional
            $limit      =& $regs['limit'][0];       // optional
            $terminator =& $regs['terminator'][0];  // optional

            // prepare string for inclusion in regular expression by escapig special characters
            $string1 = str_replace('(', "\(", $string1);
            $string1 = str_replace(')', "\)", $string1);
            $string1 = str_replace('[', "\[", $string1);
            $string1 = str_replace(']', "\]", $string1);
            $string1 = str_replace('.', "\.", $string1);
            $string1 = str_replace('?', "\?", $string1);

            // build replacement string
            if (empty($limit) AND empty($distinct)) {
                $string2 = "$function($value) OVER (ORDER BY $orderby) FROM $from WHERE $where $terminator";
            } else {
                $string2 = "DISTINCT $function($value) OVER (ORDER BY $orderby) FROM $from WHERE $where $terminator";
            } // if

            $output  = preg_replace("/$string1/iu", $string2, $output, 1);

            $str_len = strlen($string2);
            $offset = $str_start+$str_len;  // next scan starts after this string
        } // while

        return $output;

    } // adjustFirstValue

    // ****************************************************************************
    function adjustGroupBy ($select_str, $group_str, $sort_str)
    // ensure GROUP_BY contains every field in the SELECT string, plus every field
    // in the ORDER_BY string.
    {
        if (preg_match('/WITH ROLLUP/i', $group_str, $regs)) {
            // this is not recognised, so remove it
        	$group_str = str_replace($regs[0], '', $group_str);
            $with_rollup = ' WITH ROLLUP';
        } else {
            $with_rollup = null;
        } // if

        // turn $group_str into an array (delimiter is ',' followed by zero or more spaces)
        $group_array_in  = preg_split('/,\s*/', $group_str);
        $group_array_in  = array_map('trim', $group_array_in);
        $group_array_out = array();

        $alias_names = extractAliasNames($select_str);

        $pattern1 = <<<END_OF_REGEX
/
^                     # begins with
coalesce\s*\(\s*      # COALESCE(
(?<field>\w+(\.\w+)?) # word OR word.word
/imsx
END_OF_REGEX;
        $pattern2 = <<<END_OF_REGEX
/
^                                       # begins with
coalesce\s*\(\s*\(SELECT\s+.+           # 'COALESCE((SELECT ...'
,\s*(?<default>[0-9a-z]+(\.\w+)?)\)\s*  # ', <default>)' (number or word[.word])
AS\s+(?<alias>\w+)                      # 'AS <alias>'
/imsx
END_OF_REGEX;

        list($field_alias, $field_orig) = extractFieldNamesIndexed ($select_str);
        foreach ($group_array_in as $ix => $fieldname) {
            if (empty($alias_names[$fieldname])) {
                $group_array_out[] = $fieldname;  // does not have an alias, so use as-is
            } else {
                $group_array_out[] = $alias_names[$fieldname];
            } // if
        } // foreach

        if (!empty($with_rollup)) {
            // cannot merge sort_str with $group_str
        } else {
            if (!empty($sort_str)) {
        	    // turn $sort_str into an array
                //$sort_array = preg_split('/, */', $sort_str);
                $sort_array = extractOrderBy($sort_str);
                $sort_array = array_map('trim', $sort_array);
                foreach ($sort_array as $fieldname) {
                    $ix = array_search($fieldname, $field_alias);
                    if ($ix !== false) {
                	    // check that this is not an alias name
                	    if ($fieldname == $field_orig[$ix]) {
                	        if (!in_array($fieldname, $group_array_out)) {
                			    $group_array_out[] = $fieldname;
                		    } // if
                	    } // if
                    } else {
                	    if (!in_array($fieldname, $group_array_out)) {
            			    $group_array_out[] = $fieldname;
            		    } // if
                    } // if
                } // foreach
            } //  if
        } // if

        // convert amended array back into a string
        if (!empty($with_rollup)) {
            $group_str = 'ROLLUP('.implode(', ', $group_array_out).')';
        } else {
            $group_str = implode(', ', $group_array_out);
        } // if

        return $group_str;

    } // adjustGroupBy

    // ****************************************************************************
    function adjustGroupBy2 ($group_array, $select_str)
    // for some reason the statement 'COALESCE((SELECT x FROM y WHERE a=b), word) AS alias)'
    // requires that 'b' from the WHERE clause be added to the GROUP BY string
    // (but not if it is a literal))
    {
        // extract the contents of the WHERE clause
        if (preg_match('/WHERE\s+(?<where>.+?)\)/imsx', $select_str, $regs)) {
            $where_array = where2indexedArray($regs['where']);
            foreach ($where_array as $string) {
                if (preg_match('/^(AND|OR)$/i', trim($string))) {
                    // this separates one string from another, so ignore it
                } else {
                    list($fieldname, $operator, $fieldvalue) = splitNameOperatorValue($string);
                    if (preg_match("/^'.*'$/", $fieldvalue)) {
                        // this is a quoted string, so ignore it
                    } else {
                        if (!in_array($fieldvalue, $group_array)) {
                            $group_array[] = $fieldvalue;  // not there yet, so add it now
                        } // if
                    } // if
                } // if
            } // foreach

        } // if

        return $group_array;

    } // adjustGroupBy2

    // ****************************************************************************
    function adjustHaving ($select_str, $from_str, $where_str, $group_str, $having_str, $sort_str, $union=null)
    // make 'SELECT ... FROM ... WHERE ...' into a subquery so that the HAVING clause can
    // become the WHERE clause of the outer query.
    // This is because the HAVING clause cannot reference an expression by its alias name.
    {
        // Replace TRUE/FALSE to 1/0.
        $search  = array('/=[ ]*TRUE/iu', '/=[ ]*FALSE/iu');
        $replace = array( '=1',            '=0');
        $having_str = preg_replace($search, $replace, $having_str);

        $orderby_array = array();
        if (!empty($sort_str)) {
            $orderby_array = extractOrderBy($sort_str);
        } // if

        // turn select string into an associative array of 'alias=expression' pairs
        $select_array = extractAliasNames($select_str);
        // convert to 'expression=alias'
        $select_array = array_flip($select_array);

        foreach ($orderby_array as $ix => &$fieldname) {
            if (array_key_exists($fieldname, $select_array)) {
                // replace expression with alias
                $orderby_array[$ix] = $select_array[$fieldname];
            } // if
        } // foreach

        if (!empty($orderby_array)) {
            $sort_str = implode(', ', $orderby_array);
            $sort_str = unqualifyOrderBy($sort_str);
        } // if

        // put current query into a subqery
        $subquery   = "    SELECT $select_str\n    FROM $from_str $where_str $group_str";

        if (!empty($union)) {
            $union     = str_replace("\'", "''", $union);
            $subquery .= "\nUNION ALL\n$union";
            $union     = '';
        } // if

        $select_str = '*';
        $from_str   = "(\n$subquery\n) AS y";
        $having_str = unqualifyWhere($having_str, '*', true);  // unqualify any values as well
        $having_str = $this->adjustWhere($having_str, false);
        $where_str  = "\nWHERE $having_str";
        $having_str = '';
        $group_str  = '';

        return array($select_str, $from_str, $where_str, $group_str, $having_str, $sort_str);

    } // adjustHaving

    // ****************************************************************************
    function adjustOrderBy ($orderby_str, $select_str, $from_str, $group_str, $derived_table=null)
    // adjust for differences between MySQL and SQL Server.
    {
        // Replace 'SUBSTR(...)' with 'SUBSTRING(...)'.
        $orderby_str  = preg_replace('/\bSUBSTR(?!ING)\b/iu', 'SUBSTRING', $orderby_str);

        //$orderby_array = explode(',', $orderby_str);
        //$orderby_array = array_map("trim", $orderby_array);
        $orderby_array = extractOrderby($orderby_str);

        // turn select string into an associative array of 'fieldname=expression' pairs
        $select_array = extractFieldNamesAssoc($select_str);
        // obtain a 2nd array where 'fieldname' is an unqualified fieldname
        list($x1, $x2, $select_array_unq) = extractFieldNamesIndexed($select_str);
        // turn select string into an associative array of 'alias=expression' pairs
        $alias_array = extractAliasNames($select_str);

        // if $select_str contains nothing but aggregates then ORDERBY must be empty
        $aggr_count = 0;
        foreach ($alias_array as $expression) {
            if (preg_match('/\b(AVG|MIN|MAX|SUM)\b/xims', $expression, $regs)) {
                $aggr_count++;
            } // if
        } // foreach
        if ($aggr_count > 0 AND $aggr_count == count($alias_array)) {
            return null;
        } // if

        $alias_array = array_flip($alias_array);  // convert to 'expression=alias'

        if (count($select_array) == 1 AND !is_bool($group_str)) {
            // select array contains a single column name
            $fieldname  = key($select_array);
            if ($fieldname == '*') {
                // indicates all available columns, so skip the next bit
            } else {
                $expression = $select_array[$fieldname];
                if (preg_match('/^(AVG|COUNT|MIN|MAX|SUM)\b/xims', $expression, $regs)) {
                    if (empty($group_str)) {
                        // expression is an aggregation and there is no GROUP BY, so remove it
                        $orderby = null;
                        return $orderby;
                    } // if
                } // if
            } // if
        } // if

        foreach ($orderby_array as $ix => $fieldname) {
            if (array_key_exists($fieldname, $select_array)) {
                // cannot use alias name, so use expression instead
                if (preg_match("/^'.*'/", $select_array[$fieldname])) {
                    // this is a quoted string, so drop it altogether
                    unset($orderby_array[$ix]);
                } else {
                    // replace column name with expression
                    $expression = $select_array[$fieldname];
                    $orderby_array[$ix] = $expression;
                } // if
            } // if
        } // foreach

        // get this of table names which are accessed in this part of the query
        $tablenames = extractTableNames($from_str);

        $pattern1 = <<< END_OF_REGEX
/
^               # begins with
(?<table>\w+)   # <tablename>
\.              # dot
(?<column>\w+   # <columnname>
(\s+(ascending|asc|descending|desc))?   # 'ASC|DESC' (optional)
)
$               # ends with
/imsx
END_OF_REGEX;

        foreach ($orderby_array as $ix => $fieldname) {
            if (preg_match($pattern1, $fieldname, $regs)) {
                // fieldname is qualified with a table name, but does that name exist in the FROM string?
                if (array_key_exists($regs['table'], $tablenames)) {
                    // yes, so leave it alone
                } else {
                    // no, so unqualify it
                    $orderby_array[$ix] = $regs['column'];
                } // if
            } elseif (!empty($derived_table)) {
                if (array_key_exists($fieldname, $select_array_unq)) {
                    // replace unqualified name with its qualified version
                    $orderby_array[$ix] = $select_array_unq[$fieldname];
                } elseif (!array_key_exists($fieldname, $select_array_unq)) {
                    // fieldname is not qualified and is not in unqualified array, so ...
                    if (array_key_exists($fieldname, $select_array) AND !preg_match('/^\w+$/i', $select_array[$fieldname])) {
                        $test = true;  // not a simple column name, so leave it alone
                    } elseif (array_key_exists($fieldname, $alias_array)) {
                        $test = true;  // this is an complex expression, so leave it unqualified
                    } elseif (array_key_exists("$derived_table.*", $select_array)) {
                        if (preg_match('/^\w+$/', $fieldname)) {
                            // fieldname is a single word, so qualify it with id of derived table
                            $orderby_array[$ix] = "$derived_table.$fieldname";
                        } // if
                    } // if
                } // if
            } // if
        } // foreach

        // turn array back into a string
        $orderby_str = implode(', ', $orderby_array);

        $orderby_str = $this->fixReservedWords($orderby_str);

        return $orderby_str;

    } // adjustOrderBy

    // ****************************************************************************
    function adjustSelect ($input)
    // adjust for differences between MySQL and SQL Server.
    {
        $output = $input;

        do {
            $output = str_replace('  ',' ', $output, $count);  // replace double spaces with a single space
        } while ($count > 0);

        //if (str_starts_with($output, 'postal_address.address_id')) {
        if (str_starts_with($output, 'GROUP_CONCAT(prod_cat_name')) {
            //debugBreak();
        } // if

        // Replace TRUE/FALSE in CASE statements to 1/0.
        $search  = array('/THEN TRUE/iu', '/ELSE TRUE/iu', '/THEN FALSE/iu', '/ELSE FALSE/iu');
        $replace = array( 'THEN 1',        'ELSE 1',        'THEN 0',         'ELSE 0');
        $output  = preg_replace($search, $replace, $output);

        // Replace 'SUBSTR(...)' with 'SUBSTRING(...)'.
        $output  = preg_replace('/\bSUBSTR(?!ING)\b/iu', 'SUBSTRING', $output);

        // Replace 'INSTR(...)' with 'CHARINDEX(...)'.
        $output  = preg_replace('/\bINSTR\b/iu', 'CHARINDEX', $output);

        // Replace 'LENGTH(...)' with 'LEN(...)'.
        $output  = preg_replace('/\bLENGTH\(\b/iu', 'LEN', $output);

        // replace 'SELECT ... WHERE ... LIMIT 1' with 'SELECT TOP 1 ... WHERE ...'
        $pattern1 = <<< END_OF_REGEX
/
(?P<select>                     # named pattern
\bselect\s(?=[^0-9])            # 'SELECT ' (but not 'SELECT 1')')
)                               # end named pattern
.*?                             # any characters, non-greedy
(                               # start choice
(?P<limit>\bLIMIT\b\s*\d+)      # 'LIMIT n', named pattern
|                               # or
(?P<alias>\)\s*AS\s+\w+)        # ') AS alias
){1}                            # end choice (one occurrence)
/xims
END_OF_REGEX;

        $pattern2 = <<< END_OF_REGEX
/
(                               # start choice
(?P<in>\bIN\s*\(\s*)            # 'IN ('
|                               # or
(?P<notin>(?<!\bIN\b)\s*\(\s*)  # not 'IN ('
)                               # end choice
#(?P<subselect>\bselect\b)               # 'SELECT' - string contains a 2nd SELECT
(?P<subselect>\bselect\b.+?[^LIMIT\b]\)) # '(SELECT ...)' -contains a 2nd SELECT without LIMIT
/xims
END_OF_REGEX;

        $pattern3 = <<< END_OF_REGEX
/
^                             # begins with
(?P<select>                   # named pattern
\bselect\s(?=[^0-9])          # 'SELECT ' (but not 'SELECT 1')
)                             # end named pattern
(?P<body1>                    # named pattern
.*                            # any characters, non-greedy
\s*\)\s*                      # ' )
)                             # end named pattern
(?P<body2>                    # named pattern
.*                            # any characters
(?=LIMIT)                     # not ending in LIMIT
)                             # end named pattern
(?P<limit>\bLIMIT\b\s*\d+)    # 'LIMIT n', named pattern
$                             # end with
/xims
END_OF_REGEX;

        $offset = 0;
        while ($count = preg_match($pattern1, $output, $regs, PREG_OFFSET_CAPTURE, $offset)) {
            $string1 = $regs[0][0];
            $start   = $regs[0][1];
            $select  = $regs['select'][0];
            $skip = false;
            if (!empty($regs['limit'][0])) {
                if (preg_match($pattern2, $string1, $regs1, PREG_OFFSET_CAPTURE, $offset+1)) {
                    if (!empty($regs1['in'][0])) {
                        $true = 1;  // continue
                    } elseif (!preg_match('/\bLIMIT\b/i', $regs1['subselect'][0])) {
                        $true = 1;  // subselect does NOT contain LIMIT, so continue
                    } else {
                        // string contains a 2nd 'select, so restart from there
                        $offset = $start + $regs1['subselect'][1];
                        $skip = true;
                    } // if
                } // if
            } // if
            if ($skip === true) {
                // skip the next bit
            } else {
                // look for 'LIMIT 1' and replace with 'TOP 1'
                if (!empty($regs['limit'][0])) {
                    //$select  = substr($select, 0, -1);  // strip last character
                    $digits  = trim(substr($regs['limit'][0], 5));
                    $string2 = str_replace($regs['limit'][0], '', $string1);  // remove LIMIT
                    $string3 = substr_replace($string2, "{$select}TOP $digits ", 0, strlen($select));
                    $output  = substr_replace($output, $string3, $start, strlen($string1));
                    $offset = $start + strlen($string3);
                } else {
                    $offset = $start + strlen($string1);  // SELECT does not end with LIMIT
                } // if
            } // if
        } // while

        if ($offset > 0) {
            // previous pattern was found, so ignore this one
        } else {
            if (preg_match($pattern3, $output, $regs)) {
                // 'SELECT ... LIMIT n'
                $select = trim($regs['select']);
                $body   = $regs['body1'];
                if (!empty($regs['body2'])) {
                    $body .= $regs['body2'];
                } // if
                $digits  = trim(substr($regs['limit'], 5));
                $output = "$select TOP $digits $body";
                return $output;
            } // if
        } // if

        // ********************************************************************
        if (str_contains($output, 'GROUP_CONCAT(prod_cat_name)')) {
            //debugBreak();
        } // if

        // replace: (SELECT GROUP_CONCAT(table.field ORDER BY .... SEPARATOR ',') FROM table WHERE a=b) AS alias
        // with   : (SELECT STRING_AGG(table.field, delimiter ORDER BY ...) FROM table GROUP BY group) AS alias
        // OR
        // replace: GROUP_CONCAT(table.field ORDER BY .... SEPARATOR ',') AS alias
        // with   : STRING_AGG(table.field, delimiter ORDER BY ...) AS alias
        $pattern1 = <<< END_OF_REGEX
/
((?<select>\(SELECT[ ]))?          # '(SELECT ' (optional)
GROUP_CONCAT[ ]*\(                 # 'GROUP_CONCAT('
((?<distinct>[ ]*DISTINCT[ ]+))?   # 'DISTINCT ' (optional)
(?<column>(.+?)){1}                # '[table.]column' or 'CONCAT(...)'
[ ]+ORDER[ ]BY[ ]{1}               # 'ORDER BY ...'
(?<orderby>.+?)
[ ]+SEPARATOR[ ]*                  # 'SEPARATOR ...'
(?<separator>.+?)
((\)[ ]+FROM[ ]+){1}               # 'FROM ...'
(?<from>.+?))?
(\)[ ]AS[ ]){1}                    # ') AS alias'
(?<alias>\w+)
/xims
END_OF_REGEX;

//        $pattern2 = <<< END_OF_REGEX
///
//(                          # start choice
//  \w+\.\w+                 # word.word
//  |
//  \w+                      # word
//  |                        # or
//  '(([^\\\']*(\\\.)?)*)'   # quoted string
//  |                        # or
//  [ ]*,[ ]*                # [space]comma[space]
//)                          # end choice
///xims
//END_OF_REGEX;

        $offset = 0;
        while ($count = preg_match($pattern1, $output, $regs, PREG_OFFSET_CAPTURE, $offset)) {
            $string1   = $regs[0][0];
            $str_start = $regs[0][1];

            $column    = $regs['column'][0];
            $separator = $regs['separator'][0];
            $orderby   = $regs['orderby'][0];
            $alias     = $regs['alias'][0];
            $select    =& $regs['select'][0];    // optional
            $distinct  =& $regs['distinct'][0];  // optional (will be ignored)
            $from      =& $regs['from'][0];      // optional
            if (!empty($from)) {
                $from = "FROM $from";
            } // if

            // prepare string for inclusion in regular expresion by escapig special characters
            $string1 = str_replace('(', "\(", $string1);
            $string1 = str_replace(')', "\)", $string1);
            $string1 = str_replace('[', "\[", $string1);
            $string1 = str_replace(']', "\]", $string1);
            $string1 = str_replace('.', "\.", $string1);
            $string1 = str_replace('?', "\?", $string1);

            if (!empty($select)) {
                $string2 = "(SELECT STRING_AGG($column, $separator) WITHIN GROUP (ORDER BY $orderby) $from) AS $alias";
            } else {
                $string2 = " STRING_AGG($column, $separator) WITHIN GROUP (ORDER BY $orderby) AS $alias";
            } // if
            $output  = preg_replace("/$string1/iu", $string2, $output, 1);

            $str_len = strlen($string2);
            $offset = $str_start+$str_len;  // next scan starts after this string
        } // while

        // ********************************************************************

        //$output = $this->adjustCONCAT($output);

        // ********************************************************************

        // replace 'CONCAT_WS(',', A, B, C)' with "(COALESCE(A, '') + COALESCE(', ' + B, '') + COALESCE(', ' + C, ''))".
        $pattern3 = <<< END_OF_REGEX
/
(?<concat>concat_ws[ ]*\()   # 'concat_ws('
(                            # start choice
 \w+[ ]*\([^\(\)]*\)         # 'FUNC(...)'
 |
 (?<separator>'(([^\\\']*(\\\.)?)*)')      # quoted string
 |
 \w+(\.\w+)?                 # 'word' or 'word.word'
 |
 ,                           # comma
 |
 [ ]*                        # 0 or more spaces
)                            # end choice
*                            # 0 or more times
[ ]*                         # 0 or more spaces
/xims
END_OF_REGEX;

        if (str_starts_with($output, 'postal_address.address_id')) {
            //debugBreak();
        } // if
        $offset = 0;
        while ($count = preg_match($pattern3, $output, $regs, PREG_OFFSET_CAPTURE, $offset)) {
            $length    = strlen($regs[0][0]);
            $concat    = $regs['concat'];
            $separator = $regs['separator'][0];
            $field_string = substr($regs[0][0], strlen($regs['concat'][0])+strlen($separator)+1);
            $field_array  = explode(',', $field_string);
            $new_string = '(COALESCE('.array_shift($field_array).", '')";
            while (!empty($field_array)){
                $new_string .= " + COALESCE($separator + " .array_shift($field_array).", '')";
            } // while
            $output = substr_replace($output, $new_string, $regs['concat'][1], $length);
            $offset = $offset +$regs[0][1];
        } // while

        // ********************************************************************

        // Replace 'TRUNCATE(number, length)' with 'ROUND(number, length, 1)'.
        $pattern1 = <<<EOD
/
TRUNCATE[ ]*\(     # begins with 'TRUNCATE('
(?<number>.+?)     # number
,                  # ','
(?<length>.+?)     # length
\)                 # ends with ')'
/imsx
EOD;
        $offset = 0;
        while (preg_match($pattern1, $output, $regs, PREG_OFFSET_CAPTURE, $offset)) {
            $search  = $regs[0][0];
            $number  = trim($regs['number'][0]);
            $length  = trim($regs['length'][0]);
            $replace = "ROUND($number, $length, 1)";
            $output = str_replace($search, $replace, $output);
            $offset += strlen($search);
        } // while

        // ********************************************************************
        // look for the following date expressions:
        // 'DATE_ADD(field1 - INTERVAL $field2 unit)' to 'DATEADD(unit, +$field2, $field1)'

        $pattern1 = <<<EOD
/
\bDATE_ADD\b[ ]*        # function name
\(                      # opening '('
(?<field1>\w+(\(\))?)   # field1 or function()
,[ ]*INTERVAL[ ]+
(?<field2>\w+)          # field2
[ ]+
(?<unit>\w+)            # unit
[ ]*
\)                      # closing ')'
/imsx
EOD;
        $offset = 0;
        while (preg_match($pattern1, $output, $regs, PREG_OFFSET_CAPTURE, $offset)) {
            $search  = $regs[0][0];
            $field1 = $regs['field1'][0];
            $field2 = $regs['field2'][0];
            $unit   = $regs['unit'][0];
            $replace = "DATEADD($unit, $field2, $field1)";
            $output = str_replace($search, $replace, $output);
            $offset += strlen($search);
        } // while

        // 'DATE_SUB(field1 - INTERVAL $field2 unit)' to 'DATEADD(unit, -$field2, $field1)'
        $pattern1 = <<<EOD
/
\bDATE_SUB\b[ ]*        # function name
\(                      # opening '('
(?<field1>\w+(\(\))?)   # field1 or function()
,[ ]*INTERVAL[ ]+
(?<field2>\w+)          # field2
[ ]+
(?<unit>\w+)            # unit
[ ]*
\)                      # closing ')'
/imsx
EOD;
        $offset = 0;
        while (preg_match($pattern1, $output, $regs, PREG_OFFSET_CAPTURE, $offset)) {
            $search  = $regs[0][0];
            $field1 = $regs['field1'][0];
            $field2 = $regs['field2'][0];
            $unit   = $regs['unit'][0];
            $replace = "DATEADD($unit, $field2*-1, $field1)";
            $output = str_replace($search, $replace, $output);
            $offset += strlen($search);
        } // while

        // 'DATEDIFF(date1, date2)' to 'DATEDIFF(day, date2, date1)'
        // each date may be a field, a function or a literal
        $pattern1 = <<<EOD
/
\bDATEDIFF\b[ ]*        # function name
\(                      # opening '('
(?<field1>              # start named pattern
(                       # start choice
'(([^\\\']*(\\\.)?)*)'  # string literal
|                       # or
\w+\(\w+,[ ]*\w+\)      # word(word,word)
|                       # or
\w+\(\w+\)              # word(word)
|                       # or
\w+\(\)                 # word()
|                       # or
\w+                     # word
)                       # end choice
)                       # end named pattern
[ ]*,[ ]*               # comma separator
(?<field2>              # start named pattern
(                       # start choice
'(([^\\\']*(\\\.)?)*)'  # string literal
|                       # or
\w+\(\w+,[ ]*\w+\)      # word(word,word)
|                       # or
\w+\(\w+\)              # word(word)
|                       # or
\w+\(\)                 # word()
|                       # or
\w+                     # word
)                       # end choice
)                       # end named pattern
[ ]*
\)                      # closing ')'
/imsx
EOD;
        $offset = 0;
        while (preg_match($pattern1, $output, $regs, PREG_OFFSET_CAPTURE, $offset)) {
            $search  = $regs[0][0];
            $field1 = $regs['field1'][0];
            $field2 = $regs['field2'][0];
            $replace = "DATEDIFF(day, $field2, $field1)";
            //$output = str_replace($search, $replace, $output);
            //$offset += strlen($search);
            $output = substr_replace($output, $replace, $regs[0][1], strlen($search));
            $offset = $regs[0][1] + strlen($replace);
        } // while

        // ********************************************************************

        // look for any '(expression) AS alias' clauses
        //if (str_starts_with($output, 'party_relationship.party_id_1')) {
        //if (str_starts_with($output, 'postal_address.address_id,')) {
        if (str_ends_with($output, 'relationship_priority.sort_seq')) {
            //debugBreak();
        } // if
        $alias_names = extractAliasNames($output);
        $test_str = $this->replaceAliasWithExpression($output, $alias_names);
        if (!empty($test_str)) {
            $output = $test_str;
        } // if

        // rebuild alias names in case any have changed
        $alias_names = extractAliasNames($output);
        $this->alias_names = $alias_names;  // save for adjustFrom() and adjustWhere() methods

        // ********************************************************************

        $output = $this->fixReservedWords($output);

        return $output;

    } // adjustSelect

    // ****************************************************************************
    function replaceAliasWithExpression ($string, $alias_names, $operator=null)
    // search $string and replace any '=alias' with '=expression'
    {
        if (empty($alias_names) OR !is_array($alias_names)) {
            return false;
        } // if

        $alias_names_wrk = $alias_names;  // make a working copy

        foreach ($alias_names as $alias => $expression) {
            $pattern3 = <<< END_OF_REGEX
/
(                                # start choice
(?<qual1>\w+\.$alias\b)          # 'word.alias' (to be ignored)
|                                # or
(?<qual2>\b$alias\.\w+)          # 'alias.word' (to be ignored)
|                                # or
(?<func1>\s*\(\s*$alias\b.*?\))  # 'word(alias)' (to be ignored)
|                                # or
(?<func2>\b$alias\s*\()          # 'alias(word)' (to be ignored)
)                                # end choice
/xims
END_OF_REGEX;
            // remove this entry from the working copy
            unset($alias_names_wrk[$alias]);
            // now process all following entries in the working copy
            foreach ($alias_names_wrk as $alias_wrk => $expression_wrk) {
                $found  = false;
                $offset = 0;
                if (preg_match($pattern3, $expression_wrk, $regs3)) {
                    $found = false;
                } elseif (preg_match("/\b$alias\b/i", $expression_wrk, $regs4, PREG_OFFSET_CAPTURE, $offset)) {
                    $found  = true;
                    $found_string = $regs4[0][0];
                    $found_offset = $regs4[0][1];
                    if ($this->_is_within_quoted_string($alias, $expression_wrk, $offset, $found_offset)) {
                        //$offset = $found_offset + strlen($found_string);
                        $found_string = null;
                        $found_offset = null;
                    } // if
                    if (!empty($found_string) AND $found_offset > 0) {
                        // replace any expressions containing this alias with the expression for the alias
                        $replace = str_replace($alias, $expression, $expression_wrk);
                        $offset = 0;

                        $count = substr_count($string, $expression_wrk, $offset);
                        $string = str_replace($expression_wrk, $replace, $string, $count);
                    } // if
                } // if
            } // foreach
        } // foreach

        return $string;

        // **** OLD CODE *********************************************************
//        foreach ($alias_names as $alias => $expression) {
//            $pause = $alias;
//            $pattern1 = <<< END_OF_REGEX
///
//'(([^\\\']*(\\\.)?)*)'     # quoted string
///xims
//END_OF_REGEX;
//            $pattern2 = <<< END_OF_REGEX
///
//( # start choice
//(?P<function>\bCOALESCE\b\s*\()                  # possible 'function(..)'
//| # or
//(?P<aggregate>\b(AVG|COUNT|MIN|MAX|SUM)\b\s*\()  # possible 'aggregate(..)'
//){1} # end choice
//(?P<column>(?<!\.)\b$alias\b)                    # $alias, but not '.$alias'
///xims
//END_OF_REGEX;
//            $pattern3 = <<< END_OF_REGEX
///
//(                                # start choice
//(?<qual1>\w+\.$alias\b)          # 'word.alias' (to be ignored)
//|                                # or
//(?<qual2>\b$alias\.\w+)          # 'alias.word' (to be ignored)
//|                                # or
//(?<func1>\s*\(\s*$alias\b.*?\))  # 'word(alias)' (to be ignored)
//|                                # or
//(?<func2>\b$alias\s*\()          # 'alias(word)' (to be ignored)
//)                                # end choice
///xims
//END_OF_REGEX;
//           $pattern4 = <<< END_OF_REGEX
//~
//(?<operator>(!=|=|<>|<=|<|>=|>|\+|-|\*|/)) # operator
//[ ]*
//(?<column>\b$alias\b)                      # 'alias' on its own (to be updated)
//~xims
//END_OF_REGEX;
//            $pattern5 = <<< END_OF_REGEX
//~
//(?<column>\b$alias\b)                      # 'alias' on its own (to be updated)
//[ ]*
//(?<operator>(!=|=|<>|<=|<|>=|>|\+|-|\*|/)) # operator
//~xims
//END_OF_REGEX;
//
//            $offset        = 0;
//            // find out where this alias is first defined
//            if (!preg_match("/AS[ ]+\b$alias\b/i", $string, $regs, PREG_OFFSET_CAPTURE, $offset)) {
//                $not_found = true;  // this should NOT happen !!!
//            } else {
//                $found_string = $regs[0][0];
//                $found_offset = $regs[0][1];
//                // start further searches AFTER this point
//                $offset    = $found_offset + strlen($found_string);
//                do {
//                    $found        = false;
//                    $found_string = null;
//                    $found_offset = null;
//                    if (preg_match($pattern2, $string, $regs2, PREG_OFFSET_CAPTURE, $offset)) {
//                        $found = true;
//                        $found_string = $regs2['column'][0];
//                        $found_offset = $regs2['column'][1];
//                        if ($this->_is_within_quoted_string($alias, $string, $offset, $found_offset)) {
//                            $offset = $found_offset + strlen($found_string);
//                            $found_string = null;
//                            $found_offset = null;
//                        } // if
//                    } elseif (preg_match($pattern3, $string, $regs3, PREG_OFFSET_CAPTURE, $offset)) {
//                        $found = true;
//                        $offset += strlen($regs3[0][0]) + $regs3[0][1];
//
//                    } elseif (preg_match($pattern4, $string, $regs4, PREG_OFFSET_CAPTURE, $offset)) {
//                        $found = true;
//                        if (!empty($regs4['column']) AND !empty($regs4['operator'])) {
//                            if (empty($operator) OR (!empty($operator) AND $operator == $regs4['operator'])) {
//                                $found_string = $regs4['column'][0];
//                                $found_offset = $regs4['column'][1];
//                                if ($this->_is_within_quoted_string($alias, $string, $offset, $found_offset)) {
//                                    $offset = $found_offset + strlen($found_string);
//                                    $found_string = null;
//                                    $found_offset = null;
//                                } // if
//                            } // if
//                        } // if
//                    } elseif (preg_match($pattern5, $string, $regs5, PREG_OFFSET_CAPTURE, $offset)) {
//                        $found = true;
//                        if (!empty($regs5['column']) AND !empty($regs5['operator'])) {
//                            if (empty($operator) OR (!empty($operator) AND $operator == $regs5['operator'])) {
//                                $found_string = $regs5['column'][0];
//                                $found_offset = $regs5['column'][1];
//                                if ($this->_is_within_quoted_string($alias, $string, $offset, $found_offset)) {
//                                    // do not replace this name with the expression
//                                    $offset = $found_offset + strlen($found_string);
//                                    $found_string = null;
//                                    $found_offset = null;
//                                } // if
//                            } // if
//                        } // if
//                    } // if
//                    if (!empty($found_string) AND $found_offset > 0) {
//                        // relpace this occurrence of $alias with $expression
//                        $string = substr_replace($string, $expression, $found_offset, strlen($found_string));
//                        $offset += strlen($expression);
//                        $update = true;
//                    } // if
//                } while ($found == true);
//            } // if
//        } // foreach
//
//        if (is_True($update)) {
//            return $string;
//        } else {
//            return false;
//        } // if

    } // replaceAliasWithExpression

    // ****************************************************************************
    function _is_within_quoted_string ($alias, $string, $offset, $found_offset)
    // find out if this occurrence of $alias within $string is within a quoted string.
    // $offset is the starting point of the search which found $alias
    // $found_offset is where this occurrence of $alias was found
    {
        $pattern1 = <<< END_OF_REGEX
/
'(([^\\\']*(\\\.)?)*)'     # quoted string
/xims
END_OF_REGEX;

        $alias_end = strlen($alias) + $found_offset;
        $quoted_offset = $offset;
        // look for a quoted string which starts before and ends after this occurrence of $alias
        while (preg_match($pattern1, $string, $regs1, PREG_OFFSET_CAPTURE, $quoted_offset)) {
            $quoted_string = $regs1[0][0];
            $quoted_start  = $regs1[0][1];
            $quoted_end    = $quoted_start + strlen($quoted_string);
            if ($quoted_start > $found_offset) {
                return false;  // gone past
            } elseif ($quoted_end > $alias_end) {
                return true;  // this quoted string encloses the found string
            } // if
            $quoted_offset = $quoted_end;
        } // endwhile

        return false;

    } // _is_within_quoted_string

    // ****************************************************************************
    function fixReservedWords ($input)
    // find any reserved words and enclose them in double quotes
    {
        $pattern = <<<EOD
/
(\b)    # a word bounday
(?<!('|\[))(percent|yield|view)   # reserved words (not prefixed with single quote or '[')
(\b)    # a word boundary
/imsx
EOD;
        $output = '';

        while (preg_match($pattern, $input, $regs, PREG_OFFSET_CAPTURE)) {
            foreach ($regs as $ix => $found) {
                if ($ix == 0) {
                    $word  = $found[0];
                    $start = $found[1];
                    $output .= substr($input, 0, $start);               // copy string-before
                    $input   = substr($input, $start+strlen($word));    // remove string_before
                    $output .= '['.$word.']';
                    break;
                } // if
            } // foreach
        } // if

        $output .= $input;  // append what is left of input

        return $output;

    } // fixReservedWords

    // ****************************************************************************
    function adjustTempTable ($query, &$temp_tables, &$skip_errors)
    // convert from MySQL format to SQLSRV format.
    // $temp_tables is PASSED BY REFERENCE as it may be amended.
    // $skip_errors is PASSED BY REFERENCE as it may be amended.
    {
        $pattern1 = <<< END_OF_REGEX
/
^                           # begins with
DROP[ ]TEMPORARY[ ]TABLE[ ] # command
(IF[ ]EXISTS[ ])?           # optional
(?<tblname>\w*)             # table name
/xism
END_OF_REGEX;

        $pattern2 = <<< END_OF_REGEX
/
^                                # begins with
CREATE\s+TEMPORARY\s+TABLE\s+    # command
(?<tblname>\w+)                  # table name
(                                # start choice
#  \s*((?<colspecs>.+\)))         # (column specs)
#|                                # or
#  \s+LIKE[ ]+(?<tbllike>[^;]*)   # source table name
#|                                # or
  (\s*AS\s*\()?                  # optional 'AS ('
  \s*SELECT\s+(?<selectlist>.+)  # SELECT ...
  \s*FROM\s+(?<sourcetbl>\w+)    # FROM ...
  \s*((?<other>.+[^;]))?         # other (up to ';'
  (\s*\))?                       # optional ')'
)                                # end choice
/xism
END_OF_REGEX;

        $pattern3 = <<< END_OF_REGEX
/
^                               # begins with
INSERT[ ]+IGNORE[ ]+INTO[ ]+    # command
/xism
END_OF_REGEX;

        if (preg_match($pattern1, $query, $regs)) {
            //$query = "DROP TABLE #{$regs['tblname']}";
            $query = "IF OBJECT_ID('tempdb..#{$regs['tblname']}') IS NOT NULL DROP TABLE #{$regs['tblname']}";
            $temp_tables[$regs['tblname']] = $regs['tblname'];

        } elseif (preg_match($pattern2, $query, $regs)) {
            $top = null;
            if (!empty($regs['other'])) {
                if (preg_match('/(?<limit>limit[ ]+\d+)/i', $regs['other'], $regs2)) {
                    $limit = $regs2['limit'];
                    $regs['other'] = substr($regs['other'], 0, -strlen($limit));
                    $top = preg_replace('/limit/i', 'TOP', $limit);
                } // if
            } // if
            if (!empty($regs['selectlist'])) {
                $regs['selectlist'] = $this->adjustCAST($regs['selectlist']);
                $query = "SELECT {$top} {$regs['selectlist']} INTO {$regs['tblname']} FROM {$regs['sourcetbl']}";
                if (!empty($regs['other'])) {
                    $query .= " {$regs['other']}";
                } // if
            } // if
            $temp_tables[$regs['tblname']] = $regs['tblname'];

        } else {
            if (preg_match($pattern3, $query, $regs)) {
                $query = str_replace($regs[0], 'INSERT INTO ', $query);  // remove the word 'IGNORE'
            } // if
        } // if

        // insert '#' in front of each temp table name (unless it already has a '#')
        foreach ($temp_tables as $tblname) {
            $query = preg_replace("/\b(?<!#){$tblname}\b/i", "#{$tblname}", $query);
        } // foreach
        $skip_errors[] = 3604;  // duplicate key error

        return $query;

    } // adjustTempTable

    // ****************************************************************************
    function adjustWhere ($string_in, $replace_alias=true)
    // certain MySQL expressions have to be converted as they are not valid, such as:
    // 'DATE_ADD(field1 - INTERVAL $field2 unit)' to 'DATEADD(unit, +$field2, $field1)'
    // 'DATE_SUB(field1 - INTERVAL $field2 unit)' to 'DATEADD(unit, -$field2, $field1)'
    // 'NOW()' to 'SYSDATETIME()'
    // [\'] (escaped quote) must be changed to [''] (double quote)
    {
        $string_out = $string_in;

        $modified = false;

        $array = where2indexedArray($string_out);   // convert string into indexed array

        $pattern1 = <<< END_OF_REGEX
/
^               # begins with
(               # start choice
 \) OR \(       # ') OR ('
 |
 \) OR'         # ') OR'
 |
 OR \(          # 'OR ('
 |
 OR             # 'OR'
 |
 \) AND \(      # ') AND ('
 |
 \) AND         # ') AND'
 |
 AND \(         # 'AND ('
 |
 AND            # 'AND'
 |
 (\()+          # one or more '('
 |
 (\))+          # one or more ')'
)               # end choice
$               # ends with
/xi
END_OF_REGEX;

$pattern2 = <<<EOD
/
\bDATE_ADD\b[ ]*        # function name
\(                      # opening '('
(?<field1>\w+(\(\))?)   # field1 or function()
,[ ]*INTERVAL[ ]+
(?<field2>\w+)          # field2
[ ]+
(?<unit>\w+)            # unit
[ ]*
\)                      # closing ')'
/imsx
EOD;

$pattern3 = <<<EOD
/
\bDATE_SUB\b[ ]*        # function name
\(                      # opening '('
(?<field1>\w+(\(\))?)   # field1 or function()
,[ ]*INTERVAL[ ]+
(?<field2>\w+)          # field2
[ ]+
(?<unit>\w+)            # unit
[ ]*
\)                      # closing ')'
/imsx
EOD;

        foreach ($array as $key => $value) {
            if (preg_match($pattern1, $value, $regs)) {
                $ignore = true;  // ignore this
            } elseif (preg_match($pattern2, $value, $regs)) {
                $search = $regs[0];
                $field1 = $regs['field1'];
                $field2 = $regs['field2'];
                $unit   = $regs['unit'];
                $replace = "DATEADD($unit, $field2, $field1)";
                $array[$key] = str_replace($search, $replace, $value);
                $modified = true;

            } elseif (preg_match($pattern3, $value, $regs)) {
                $search = $regs[0];
                $field1 = $regs['field1'];
                $field2 = $regs['field2'];
                $unit   = $regs['unit'];
                $replace = "DATEADD($unit, $field2*-1, $field1)";
                $array[$key] = str_replace($search, $replace, $value);
                $modified = true;

            } else {
                list($fieldname, $operator, $fieldvalue) = splitNameOperatorValue($value);
                $fieldvalue = trim($fieldvalue);
                if (preg_match("#\\\'#", $fieldvalue)) {
                    // replace [\'] (escaped quote) with [''] (double quote)
                    $fieldvalue = str_replace("\'", "''", $fieldvalue);
                    $array[$key] = $fieldname.$operator.$fieldvalue;
                	$modified = true;

//                } elseif (preg_match('/date_add/i', $fieldname)) {
//                    // contains 'DATE_ADD(field1, INTERVAL field2 unit)', so extract 'field1, field2'
//                    preg_match('/(?<=\().+(?=\))/', $fieldname, $regs);
//                    list($field1, $interval, $field2, $unit) = preg_split("/[\s,]+/", $regs[0]);
//                    $field3 = $fieldvalue;
//                    // replace with 'DATEADD(unit, field2, field1) op $field3'
//                    $fieldname = "DATEADD($unit, $field2, $field1)";
//                    $array[$key] = $fieldname.' '.$operator.' '.trim($field3);
//                    $modified = true;
//
//                } elseif (preg_match('/^date_add/i', $fieldvalue)) {
//                    // contains 'DATE_ADD(field1, INTERVAL field2 unit)', so extract 'field1, field2, unit'
//                    preg_match('/(?<=\().+(?=\))/', $fieldvalue, $regs);
//                    list($field1, $interval, $field2, $unit) = preg_split("/[\s,]+/", $regs[0]);
//                    // replace with 'fieldname op DATEADD(unit, field2, field1)'
//                    $fieldvalue = "DATEADD($unit, $field2, $field1)";
//                    $array[$key] = $fieldname.' '.$operator.' '.trim($fieldvalue);
//                    $modified = true;
//
//                } elseif (preg_match('/^date_sub/i', $fieldname)) {
//                    // contains 'DATE_SUB(field1, INTERVAL field2 unit)', so extract 'field1, field2, unit'
//                    preg_match('/(?<=\().+(?=\))/', $fieldname, $regs);
//                    list($field1, $interval, $field2, $unit) = preg_split("/[\s,]+/", $regs[0]);
//                    $field3 = $fieldvalue;
//                    // replace with 'DATEADD(unit, field2*-1, field1) op $field3'
//                    $fieldname = "DATEADD($unit, $field2*-1, $field1)";
//                    $array[$key] = $fieldname.' '.$operator.' '.trim($field3);
//                    $modified = true;
//
//                } elseif (preg_match('/^date_sub/i', $fieldvalue)) {
//                    // contains 'DATE_SUB(field1, INTERVAL field2 unit)', so extract 'field1, field2, unit'
//                    preg_match('/(?<=\().+(?=\))/', $fieldvalue, $regs);
//                    list($field1, $interval, $field2, $unit) = preg_split("/[\s,]+/", $regs[0]);
//                    // replace with 'fieldname op DATEADD(unit, field2*-1, field1)'
//                    $fieldvalue = "DATEADD($unit, $field2*-1, $field1)";
//                    $array[$key] = $fieldname.' '.$operator.' '.trim($fieldvalue);
//                    $modified = true;

                } // if
            } // if
        } // foreach

        if ($modified) {
        	$string_out = implode(' ', $array);
        } // if

        // replace 'SELECT ... WHERE ... LIMIT 1' with 'SELECT TOP 1 ... WHERE ...'
        $pattern1 = <<< END_OF_REGEX
/
(?P<select>\([ ]*select[ ][0-9]?)   # 'SELECT ' or 'SELECT 1', named pattern
.*?                                 # any characters, non-greedy
(                                   # start choice
\)[ ]AS[ ]\w+                       # ') AS x'
|                                   # or
(?P<limit>LIMIT[ ]\d+)              # 'LIMIT n', named pattern
|                                   # or
(?P<subselect>select[ ])            # nested 'SELECT ', named pattern
)                                   # end choice
/xims
END_OF_REGEX;

        $offset = 0;
        while ($count = preg_match($pattern1, $string_out, $regs, PREG_OFFSET_CAPTURE, $offset)) {
            $string1 = $regs[0][0];
            $start   = $regs[0][1];
            $prefix  = $regs['select'][0];
            if (!empty($regs['subselect'])) {
                $offset = $start + 10;  // select contains subselect, so skip over first select
            } elseif (!empty($regs['limit'])) {
                $digits  = trim(substr($regs['limit'][0], 5));
                $string2 = str_replace($regs['limit'][0], '', $string1);  // remove LIMIT
                $string3 = substr_replace($string2, "(SELECT TOP $digits ", 0, strlen($prefix));
                $string_out  = substr_replace($string_out, $string3, $start, strlen($string1));
                $offset = $start +  strlen($string3);
            } else {
                $offset = $start +  strlen($string1);  // SELECT does not end with LIMIT
            } // if
        } // while

        if (is_True($replace_alias)) {
            // search $output and replace any '=alias' with '=expression'
            foreach ($this->alias_names as $alias => $expression) {
                $pattern = "/=[ ]*{$alias}\b/imsu";
                if (preg_match($pattern, $string_out, $regs)) {
                    $string_out = preg_replace($pattern, '='.$expression.' ', $string_out, -1, $count);
                } // endwhile
            } // foreach
        } // if

        // Replace TRUE/FALSE with 1/0.
        $search  = array('/=[ ]*TRUE/iu', '/=[ ]*FALSE/iu');
        $replace = array( '=1',            '=0');
        $string_out = preg_replace($search, $replace, $string_out);

        if (preg_match('/(?<now>\bnow\(\))/i', $string_out, $regs)) {
            $string_out = preg_replace('/\bnow\(\)/i', "SYSDATETIME()", $string_out);
        } // if

        $string_out = $this->fixReservedWords($string_out);

        return $string_out;

    } // adjustWhere

    // ****************************************************************************
    function array2string ($array)
    // return an array of values (for an ARRAY datatype) as a string.
    {
        // return array as a comma-separated string inside curly braces
        $string = '{' .implode(',', $array) .'}';

        return $string;

    } // array2string

    // ****************************************************************************
    function buildKeyString ($fieldarray, $key)
    // build a string like "name1='value1' AND name2='value2'"
    // $fieldarray is an associative array of names and values
    // $key        is an indexed array of key fields
    {
        $where_array = array();

        foreach ($key as $fieldname) {
            $fieldvalue = null;
            if (array_key_exists($fieldname, $fieldarray)) {
            	$fieldvalue = $this->adjustData($fieldarray[$fieldname]);
            } else {
                $fieldvalue = '';
            } // if
            if (empty($fieldvalue)) {
                return false;  // if any component is null the whole key is treated as null
            } // if
            $where_array[$fieldname] = $fieldvalue;
        } // foreach

        $where = array2where($where_array);

        if (empty($where)) {
        	// *NO PRIMARY KEY HAS BEEN DEFINED*
        	$where = getLanguageText('sys0033');
        } // if

        return $where;

    } // buildKeyString

    // ****************************************************************************
    function commit ($dbname)
    // commit this transaction
    {
        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        $this->connect($dbname) or trigger_error('SQLSRV', E_USER_ERROR);

        $start_time = getMicroTime();
        $result = sqlsrv_commit($this->dbconnect);
        $end_time = getMicroTime();

        // write query to log file, if option is turned on
        logSqlQuery ($dbname, null, 'COMMIT', null, $start_time, $end_time);
        $this->query = '';

        if (defined('TRANSIX_NO_AUDIT') OR defined('NO_AUDIT_LOGGING')) {
        	// do nothing
	    } else {
            $auditobj = RDCsingleton::getInstance('audit_tbl');
            $result = $auditobj->close();
        } // if

        return $result;

    } // commit

    // ****************************************************************************
    function connect ($dbname='')
    // establish a connection to the database
    {
        $this->errors = array();
        $this->query  = '';
        $this->dbname = $dbname;

        $dbconn = $this->dbconnect;

        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        if (!$dbconn) {
            $start_time = getMicroTime();

            // fill in default settings
            if (!isset($this->connectionInfo['CharacterSet'])) {
            	$this->connectionInfo['CharacterSet'] = 'UTF-8';
            } // if
            if (!isset($this->connectionInfo['ReturnDatesAsStrings'])) {
            	$this->connectionInfo['ReturnDatesAsStrings'] = true;
            } // if
            if (!empty($dbname)) {
            	$this->connectionInfo['Database'] = $dbname;
            	$this->dbname                     = $dbname;
            } // if

            //$dbconn = sqlsrv_connect($this->serverName, $this->connectionInfo) or trigger_error('SQLSRV', E_USER_ERROR);
            $dbconn = sqlsrv_connect($this->serverName, $this->connectionInfo);
            if ($dbconn === false) {
                if (preg_match('/^(258)$/', $this->getErrorNo())) {
                    // timeout error, so try again
                    sleep(1);  // pause for 1 second
                    $dbconn = sqlsrv_connect($this->serverName, $this->connectionInfo);
                } // if
                if ($dbconn === false) {
                    trigger_error('SQLSRV', E_USER_ERROR);
                } // if
            } // if
            if ($dbconn) {
                $this->dbconnect = $dbconn;

                $client_info = sqlsrv_client_info($dbconn);
                foreach ($client_info as $key => $value) {
                    if (empty($this->client_info)) {
                    	$this->client_info  =       $key.": ".$value;
                    } else {
                        $this->client_info .= ', ' .$key.": ".$value;
                    } // if
                } // if

                $server_info = sqlsrv_server_info($dbconn);
                foreach ($server_info as $key => $value) {
                    if (empty($this->server_info)) {
                        $this->server_info  =       $key.": ".$value;
                    } else {
                        $this->server_info .= ', ' .$key.": ".$value;
                    } // if
                } // if

                $this->query = "SET IMPLICIT_TRANSACTIONS OFF";
                $result = sqlsrv_query($dbconn, $this->query) or trigger_error('SQLSRV', E_USER_ERROR);

                $this->query = '';

                // write query to log file, if option is turned on
                $end_time = getMicroTime();
                logSqlQuery ($dbname, null, 'CONNECT', null, $start_time, $end_time);
            } // if
        } // if
        if (!$dbconn) {
            return FALSE;
        } // if

        //if (!empty($dbname) AND $dbname != $this->dbname_prev) {
        if (!empty($dbname)) {
            $start_time = getMicroTime();

            $this->query = "USE [$dbname]";
            $result = sqlsrv_query($dbconn, $this->query) or trigger_error('SQLSRV', E_USER_ERROR);
            // write query to log file, if option is turned on
            logSqlQuery ($dbname, null, $this->query);
            $this->query = '';
            $this->dbname = $dbname;

            // write query to log file, if option is turned on
            $end_time = getMicroTime();
            logSqlQuery ($dbname, null, $this->query, null, $start_time, $end_time);
        } // if

        $this->dbname_prev = $dbname;

        return TRUE;

    } // connect

    // ****************************************************************************
    function deleteRecord ($dbname, $tablename, $fieldarray)
    // delete the record whose primary key is contained within $fieldarray.
    {
        $this->errors = array();

        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        $this->connect($dbname) or trigger_error('SQLSRV', E_USER_ERROR);

        // build 'where' string using values for primary key
        $where = $this->buildKeyString ($fieldarray, $this->primary_key);

        if (empty($where)) return;    // nothing to delete, so exit

        // build the query string and run it
        $this->query = "DELETE FROM {$this->dbname}.{$this->dbschema}.$tablename WHERE $where";
        $start_time = getMicroTime();
        $result = sqlsrv_query($this->dbconnect, $this->query) or trigger_error('SQLSRV', E_USER_ERROR);
        $end_time = getMicroTime();

        // get count of affected rows as there may be more than one
        $this->numrows = sqlsrv_rows_affected($result);

        // write query to log file, if option is turned on
        logSqlQuery ($dbname, $tablename, $this->query, $this->numrows, $start_time, $end_time);

        if ($this->audit_logging) {
        	if (defined('TRANSIX_NO_AUDIT') OR defined('NO_AUDIT_LOGGING')) {
        		// do nothing
	        } else {
	            $auditobj = RDCsingleton::getInstance('audit_tbl');
	            // add record details to audit database
	            $auditobj->auditDelete($dbname, $tablename, $this->fieldspec, $where, $fieldarray);
	            $this->errors = array_merge($auditobj->getErrors(), $this->errors);
			} // if
        } // if

        return $fieldarray;

    } // deleteRecord

    // ****************************************************************************
    function deleteSelection ($dbname, $tablename, $selection, $from=null, $using=null)
    // delete a selection of records in a single operation.
    {
        $this->errors = array();

        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        $this->connect($dbname) or trigger_error('SQLSRV', E_USER_ERROR);

        $selection = $this->adjustWhere($selection);

        if (!empty($from) AND !empty($using)) {
            //$this->query = "DELETE FROM $from USING $using WHERE $selection";
            // multi-table delete does not work in this DBMS, so use standard delete with foreign keys
            $this->query = "DELETE FROM $tablename WHERE $selection";
        } else {
            $this->query = "DELETE FROM $tablename WHERE $selection";
        } // if
        $start_time = getMicroTime();
        $result = sqlsrv_query($this->dbconnect, $this->query) or trigger_error('SQLSRV', E_USER_ERROR);
        $end_time = getMicroTime();

        $count = sqlsrv_rows_affected($result);

        // write query to log file, if option is turned on
        logSqlQuery ($dbname, $tablename, $this->query, $count, $start_time, $end_time);

        if ($this->audit_logging) {
        	if (defined('TRANSIX_NO_AUDIT') OR defined('NO_AUDIT_LOGGING')) {
        		// do nothing
	        } else {
	            $auditobj = RDCsingleton::getInstance('audit_tbl');
	            // add record details to audit database
	            $auditobj->auditDelete($dbname, $tablename, $this->fieldspec, $selection, array());
	            $this->errors = array_merge($auditobj->getErrors(), $this->errors);
			} // if
        } // if

        return $count;

    } // deleteSelection

    // ****************************************************************************
    function fetchRow ($dbname, $result)
    // Fetch a row from the given result set (created with getData_serial() method).
    {
        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        $this->connect($dbname) or trigger_error('SQLSRV', E_USER_ERROR);

        $row   = sqlsrv_fetch_array($result, SQLSRV_FETCH_ASSOC);
        if ($row) {
        	$array = array_change_key_case($row, CASE_LOWER);
        	return $array;
        } else {
            return false;
        } // if

    } // fetchRow

    // ****************************************************************************
    function findDBVersion ($dbname)
    // return the version number for this database server.
    {
        $this->connect($dbname) or trigger_error('SQLSRV', E_USER_ERROR);

        $array = sqlsrv_server_info($this->dbconnect);

        return $array['SQLServerVersion'];

    } // findDBVersion

    // ****************************************************************************
    function free_result ($dbname, $resource)
    // release a resource created with getData_serial() method.
    {
        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        $this->connect($dbname) or trigger_error('SQLSRV', E_USER_ERROR);

        $result = sqlsrv_free_stmt($resource);

        return $result;

    } // free_result

    // ****************************************************************************
    function getCount ($dbname, $tablename, $where, $object=null)
    // get count of records that satisfy selection criteria in $where.
    {
        $this->errors = array();

        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        $this->connect($dbname) or trigger_error('SQLSRV', E_USER_ERROR);

        if (preg_match('/^(select )/ims', $where)) {
            // $where starts with 'SELECT' so use it as a complete query
            if (!empty($object) AND is_object($object)) {
                $where = $this->adjustSelect($where);  // called from parent object, so may need to be adjusted
            } // if
            $this->query = $where;
        } else {
            // does not start with 'SELECT' so it must be a 'where' clause
            if (empty($where)) {
            	$this->query = "SELECT count(*) FROM $tablename";
            } else {
                $where = $this->adjustWhere($where);
                $this->query = "SELECT count(*) FROM $tablename WHERE $where";
            } // if
        } // if

        $start_time = getMicroTime();
        $result = sqlsrv_query($this->dbconnect, $this->query);
        if ($result === false) {
            $errno = $this->getErrorNo();
            if ($errno == 1222 AND is_True($this->no_abort_on_lock_wait)) {
                $res = $this->rollback($dbname);
                $this->lock_wait_count++;
                return false;  // return and allow user to retry this transaction
            } else {
                trigger_error('SQLSRV', E_USER_ERROR);
            } // if
        } // if

        $query_data = sqlsrv_fetch_array($result, SQLSRV_FETCH_NUMERIC);

        $pattern1 = <<< END_OF_PATTERN
/
(?<derived>FROM\s*\(\s*SELECT.+FROM.+\))
/imsx
END_OF_PATTERN;
        $pattern2 = <<< END_OF_PATTERN
/
(                               # start choice
  ^WITH\s                       # starts with 'WITH '
|                               # or
  \(\s*SELECT[ ].+group by.+\)  # contains '(SELECT ... GROUP BY ...)
)                               # end choice
/imsx
END_OF_PATTERN;

        $string = $this->query;
        // check to see if query contains '..FROM(SELECT ...FROM...)'
        if (preg_match($pattern1, $this->query, $regs, PREG_OFFSET_CAPTURE)) {
            // yes it does, so strip off this part of the query
            $start  = $regs['derived'][1];
            $offset = strlen($regs['derived'][0]) + $start;
            $string = substr($string, $offset);
        } // if

        // if 'GROUP BY' was used then return the number of rows
        // (ignore GROUP BY if it is in a subselect or a CTE

        if (preg_match("/\bgroup[ ]by\b/im", $string) == true AND !preg_match($pattern2, $string)) {
            $count = sqlsrv_num_rows($result);
            if ($count === false) {  // BUG!! so have to count the rows manually
                $count = 0;
                if (!empty($query_data)) {
                    $count = 1;
                    while ($row = sqlsrv_fetch_array($result, SQLSRV_FETCH_ASSOC)) {
                        $count++;
                    } // while
                } // if
            } // if
        } else {
            if (!empty($query_data)) {
                $count = $query_data[0];
            } else {
                $count = 0;
            } // if
        } // if
        $end_time = getMicroTime();

        // write query to log file, if option is turned on
        logSqlQuery ($dbname, $tablename, $this->query, $count, $start_time, $end_time);
        $this->query = '';

        return $count;

    } // getCount

    // ****************************************************************************
    function getData ($dbname, $tablename, $where)
    // get data from a database table using optional 'where' criteria.
    // Results may be affected by $where and $pageno.
    {
        if (is_null($where)) {
            $where = '';  // set to empty string (for version 8.1)
        } // if

        $this->errors = array();

        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        $this->connect($dbname) or trigger_error('SQLSRV', E_USER_ERROR);

        $pageno         = (int)$this->pageno;
        $rows_per_page  = (int)$this->rows_per_page;
        $this->numrows  = 0;
        $this->lastpage = 0;

        $array = array();  // to be filled with database data

        // look for optional SELECT parameters, or default to all fields
        if (empty($this->sql_select)) {
            // the default is all fields
            $select_str = '*';
        } else {
            if (str_contains($this->sql_select, 'GROUP_CONCAT(prod_cat_name')) {
                //debugbreak();
            } // if
            $select_str = $this->adjustSelect($this->sql_select);
            $this->sql_select = $select_str;
//            if (empty($this->sql_groupby)) {
//                $build_groupby = false;
//                $alias_names = $this->alias_names;
//                foreach ($alias_names as $alias => $value) {
//                    if (preg_match('/STRING_AGG\(/i', $value)) {
//                        $build_groupby = true;
//                        break;
//                    } // if
//                } // foreach
//                if (is_True($build_groupby)) {
//                    $field_names = extractFieldNamesAssoc($select_str);
//                    foreach ($alias_names as $alias => $value) {
//                        unset($field_names[$alias]);
//                    } // if
//                    $this->sql_groupby = implode(', ', $field_names);
//                } // if
//            } // if
        } // if

        // use specified FROM parameters, or default to current table name
        if (empty($this->sql_from)) {
            // the default is current table
            $from_str = $tablename;
        } else {
            $from_str = $this->sql_from;
            // insert <newline> in front of every JOIN statement for readability
            $search_array  = array("/(?<!\n)LEFT JOIN/", "/(?<!\n)RIGHT JOIN/", "/(?<!\n)CROSS JOIN/", "/(?<!\n)INNER JOIN/");
            $replace_array = array(      "\nLEFT JOIN",        "\nRIGHT JOIN",        "\nCROSS JOIN",        "\nINNER JOIN");
            $from_str = preg_replace($search_array, $replace_array, $from_str);
            $from_str = $this->adjustFrom($from_str, $where);
        } // if

        // incorporate optional 'where' criteria
        $where = trim($where);
        if (empty($where)) {
            $where_str = '';
        } else {
            $where_str = "\nWHERE " .$this->adjustWhere($where);
        } // if

        if (!empty($this->sql_where_append)) {
            if (empty($where_str)) {
                $where_str = "\nWHERE {$this->sql_where_append}";
            } else {
                $where_str .= " AND {$this->sql_where_append}";
            } // if
        } // if

        // incorporate optional GROUP BY parameters
        if (!empty($this->sql_groupby)) {
            $group_str = $this->adjustGroupBy ($select_str, $this->sql_groupby, $this->sql_orderby);
            $group_str = "\nGROUP BY " .$group_str;
        } else {
            $group_str = NULL;
        } // if

        if (!empty($this->sql_CTE_name)) {
            if (!empty($this->sql_search)) {
                if (empty($where_str)) {
                    $search_str = "\nWHERE ".$this->sql_search;
                } else {
                    $search_str = " AND ".$this->sql_search;
                } // if
            } else {
                $search_str = null;
            } // if
            //if (!empty($this->sql_CTE_select)) {
            //    $this->sql_CTE_select = $this->fixReservedWords($this->sql_CTE_select);
            //} // if
            foreach ($this->sql_CTE_name as $ix => $cte_name) {
                $this->sql_CTE_name[$ix]   = $this->fixReservedWords($this->sql_CTE_name[$ix]);
                $this->sql_CTE_anchor[$ix] = $this->fixReservedWords($this->sql_CTE_anchor[$ix]);
                $this->sql_CTE_anchor[$ix] = $this->adjustSelect($this->sql_CTE_anchor[$ix]);
            } // foreach
            if (!empty($this->sql_CTE_recursive)) {
                // insert "UNION ALL" between the anchor and recursive segments
                $this->sql_CTE_recursive = "\n  UNION ALL\n\n  ".$this->sql_CTE_recursive;
            } // if
        } // if

        if (!empty($this->sql_derived_table)) {
            //$this->sql_derived_orderby = unqualifyOrderBy($this->sql_orderby) ." $this->sql_orderby_seq";
            $this->sql_derived_orderby = null;
            list($select_str, $from_str, $append_str) = $this->adjust_derived_query($select_str, $from_str, $this->sql_having, $this->sql_search, $tablename);
            $having_str        = null;
            //$this->sql_orderby = unqualifyOrderBy($this->sql_orderby);
        } elseif (!empty($this->sql_having)) {
            list($select_str, $from_str, $where_str, $group_str, $having_str, $this->sql_orderby) = $this->adjustHaving ($select_str, $from_str, $where_str, $group_str, $this->sql_having, $this->sql_orderby, $this->sql_union);
            $group_str       = null;
            $having_str      = null;
            $this->sql_union = null;
            $append_str      = null;
        } else {
            $having_str = NULL;
            $append_str = NULL;
        } // if

        // incorporate optional sort order
        if (!empty($this->sql_derived_orderby)) {
            $sort_str          = null;
        } elseif (!empty($this->sql_orderby)) {
            if (preg_match('/^(NULL)$/i', $this->sql_orderby)) {
                $sort_str = null;  // convert literal 'NULL' to actual NULL
            } else {
                $this->sql_orderby = $this->adjustOrderBy($this->sql_orderby, $select_str, $from_str, $group_str, $this->sql_derived_table);
                if (!empty($this->sql_orderby)) {
                    $pattern = '/( asc| ascending| desc| descending)$/i';
                    if (preg_match($pattern, $this->sql_orderby, $regs)) {
                        $sort_str = $this->sql_orderby;  // already contains 'asc' or 'desc'
                    } else {
                        $sort_str = "$this->sql_orderby $this->sql_orderby_seq";
                    } // if
                } else {
                    $sort_str = null;
                } // if
            } // if
        } else {
            $sort_str = implode(',', $this->primary_key);
        } // if

        if (!empty($having_str)) {
            $having_str = "\nHAVING $this->sql_having";
        } // if

        if ($rows_per_page > 0) {
            // count the rows that satisfy this query
            if (!empty($this->sql_CTE_name)) {
                $query = "WITH {$this->sql_CTE_name[0]}";

                foreach ($this->sql_CTE_anchor as $ix => $anchor) {
                    if ($ix > 0) {
                        $query .= "), {$this->sql_CTE_name[$ix]}";
                    } // if
                    $query .= "{$this->sql_CTE_anchor[$ix]}";
                    if ($ix == 0 AND !empty($this->sql_CTE_recursive)) {
                        $query .= "{$this->sql_CTE_recursive}";
                    } // if
                } // foreach

                $query .= ")\n-- end of CTE --\nSELECT count(*) FROM $from_str $where_str $search_str";

            } elseif (!empty($this->sql_union)) {
                $query = "SELECT count(*) FROM (\nSELECT $select_str \nFROM $from_str \n$where_str $group_str $having_str \nUNION ALL\n {$this->sql_union} \n) AS x $append_str";
            } else {
                $query = "SELECT count(*) FROM (\nSELECT $select_str \nFROM $from_str \n$where_str $group_str $having_str \n) AS x $append_str";
            } // if

            $this->numrows = $this->getCount($dbname, $tablename, $query);

            if ($this->numrows <= 0) {
                $this->pageno   = 0;
                $this->lastpage = 0;
                return $array;
            } // if

            // calculate the total number of pages from this query
            $this->lastpage = ceil($this->numrows/$rows_per_page);
        } else {
            $this->lastpage = 1;
        } // if

        // ensure pageno is within range
        if ($pageno < 1) {
            $pageno = 1;
        } elseif ($pageno > $this->lastpage) {
            $pageno = $this->lastpage;
        } // if
        $this->pageno = $pageno;

        $lock_str = null;
        if ($GLOBALS['transaction_has_started'] == TRUE) {
            if (is_True($this->no_read_lock)) {
                // do not apply a lock on this read
            } else {
                //if ($GLOBALS['lock_tables'] == FALSE) {
                if (empty($this->lock_tables)) {
            	    //if (empty($this->row_locks)) {
                    //    // not defined locally, but may be defined globally
                	//    $this->row_locks = $GLOBALS['lock_rows'];
                    //} // if
                    // deal with row locking (optional)
    //                switch ($this->row_locks){
    //                    case 'SH':
    //                        $lock_str = 'FOR UPDATE';
    //                        break;
    //                    case 'EX':
    //                        $lock_str = 'FOR UPDATE';
    //                        break;
    //                    default:
    //                        $count = preg_match_all("/\w+/", $from_str, $regs);
    //                        if ($count > 1) {
    //                            $lock_str = 'FOR UPDATE OF ' .$tablename;
    //                        } else {
    //                            $lock_str = 'FOR UPDATE';
    //                        } // if
    //                } // switch
                    $this->row_locks = null;
                } // if
            } // if
        } // if

        if ($rows_per_page > 0) {
            // insert code for pagination ...
            if (!empty($this->skip_offset)) {
                $min_rows = $this->skip_offset;  // use pre-calculated value
            } else {
                $min_rows = (($pageno - 1) * $rows_per_page) +1;
            } // if
            $max_rows = ($min_rows + $rows_per_page) -1;

            if (!empty($this->sql_CTE_name)) {
                if (!empty($sort_str)) {
                    $sort_str = "ORDER BY $sort_str";
                } // if
                $this->query = "WITH {$this->sql_CTE_name[0]}";

                $pattern1 = <<< END_OF_REGEX
/
ROW_NUMBER\(\)[ ]OVER[ ]\(ORDER[ ]BY[ ]
(?P<sort_str>  # pattern name
.*
)              # end pattern
\)
/imsx
END_OF_REGEX;

                foreach ($this->sql_CTE_anchor as $ix => $anchor) {
                    if ($ix > 0) {
                        $this->query .= "), {$this->sql_CTE_name[$ix]}";
                    } // if
                    $this->query .= " {$this->sql_CTE_anchor[$ix]}";
                    if (empty($sort_str)) {
                        // extract any ORDER BY string from the current
                        if (preg_match($pattern1, $this->sql_CTE_anchor[$ix], $regs)) {
                            $cte_sort_str = 'ORDER BY '.$regs['sort_str'];
                        } // if
                    } // if
                    if ($ix == 0 AND !empty($this->sql_CTE_recursive)) {
                        $this->query .= "{$this->sql_CTE_recursive}";
                    } // if
                } // foreach
                if (empty($sort_str) AND !empty($cte_sort_str)) {
                    $sort_str = $cte_sort_str;  // replace with last known ORDER BY string
                } // if

                $this->query .= ")
-- end of CTE --
SELECT * FROM (
  SELECT $select_str
  , ROW_NUMBER() OVER ($sort_str) AS rownum
  FROM $from_str
  $where_str $search_str
) AS x WHERE rownum BETWEEN $min_rows AND $max_rows";

            } elseif (!empty($this->sql_union)) {
                if (!empty($sort_str)) {
                    $sort_str = unqualifyOrderBy($sort_str);
                    $sort_str = "ORDER BY $sort_str";
                } // if
                $this->query = "SELECT * FROM ("
                             . "\n  SELECT *, ROW_NUMBER() OVER ($sort_str) AS rownum"
                             . "\n  FROM ("
                             . "\n    SELECT $select_str"
                             . "\n    FROM $from_str"
                             . "\n    $where_str $group_str $having_str"
                             . "\n    UNION ALL"
                             . "\n    {$this->sql_union}"
                             . "\n  ) AS x"
                             . "\n) AS y"
                             . "\nWHERE rownum BETWEEN $min_rows AND $max_rows $sort_str";
            } else {
                if (!empty($sort_str)) {
                    $sort_str = "ORDER BY $sort_str";
                } // if
                $this->query = "SELECT * FROM ("
                             . "\n  SELECT $select_str"
                             . "\n  , ROW_NUMBER() OVER ($sort_str) AS rownum"
                             . "\n  FROM $from_str"
                             . "\n  $where_str $group_str $having_str $append_str $lock_str"
                             . "\n) AS x WHERE rownum BETWEEN $min_rows AND $max_rows";
            } // if

        } else {
            // read all available rows
            if (!empty($this->sql_CTE_name)) {
                if (!empty($sort_str)) {
                    $sort_str = "\nORDER BY $sort_str";
                } // if
                $this->query = "WITH {$this->sql_CTE_name[0]}";

                foreach ($this->sql_CTE_anchor as $ix => $anchor) {
                    if ($ix > 0) {
                        $this->query .= "), {$this->sql_CTE_name[$ix]}";
                    } // if
                    $this->query .= " {$this->sql_CTE_anchor[$ix]}";
                    if ($ix == 0 AND !empty($this->sql_CTE_recursive)) {
                        $this->query .= "{$this->sql_CTE_recursive}";
                    } // if
                } // foreach

                $this->query .= ")
-- end of CTE --
SELECT $select_str
  FROM $from_str $where_str $group_str $having_str $sort_str";

            } elseif (!empty($this->sql_union)) {
                if (!empty($sort_str)) {
                    $sort_str = unqualifyOrderBy($sort_str);
                    $sort_str = "\nORDER BY $sort_str";
                } // if
                $this->query = "SELECT $select_str "
                             . "\nFROM $from_str "
                             . "\n$where_str $group_str $having_str"
                             . "\nUNION ALL"
                             . "\n{$this->sql_union}"
                             . "\n$sort_str";
                $this->sql_union = null;
            } else {
                if (!empty($group_str)) {
                    $sort_str = null;
                } // if
                if (!empty($append_str)) {
                    // move $sort_str to AFTER $append_str
                    $append_str .= "\nORDER BY ".unqualifyOrderBy($sort_str);
                    $sort_str    = null;
                } elseif (!empty($sort_str)) {
                    $sort_str = "\nORDER BY $sort_str";
                } // if
                $this->query = "SELECT $select_str \nFROM $from_str \n$where_str $group_str $having_str $sort_str $append_str";
            } // if
        } // if

        $start_time = getMicroTime();
        $result = sqlsrv_query($this->dbconnect, $this->query, array(), array('scrollable' => SQLSRV_CURSOR_STATIC));
        if ($result === false) {
            $errno = $this->getErrorNo();
            if ($errno == 1222 AND is_True($this->no_abort_on_lock_wait)) {
                $res = $this->rollback($dbname);
                $this->lock_wait_count++;
                return false;  // return and allow user to retry this transaction
            } else {
                trigger_error('SQLSRV', E_USER_ERROR);
            } // if
        } // if

        // convert result set into a simple associative array for each row
        while ($row = sqlsrv_fetch_array($result, SQLSRV_FETCH_ASSOC)) {
            $array[] = array_change_key_case($row, CASE_LOWER);
        } // while

        if ($rows_per_page == 0) {
            $this->numrows = sqlsrv_num_rows($result);
        } // if
        $end_time = getMicroTime();

        // write query to log file, if option is turned on
        logSqlQuery ($dbname, $tablename, $this->query, $this->numrows, $start_time, $end_time);

        sqlsrv_free_stmt($result);

        return $array;

    } // getData

    // ****************************************************************************
    function _convert_from_utf16_to_utf8 ($row, $stmt2, $fieldspec)
    // convert string fields from utf16 to utf8.
    // $row = contains a complete row as an associative array
    // $stmt2 = a 2nd resource to read each field one at a time by their index number
    {
        $row = array_change_key_case($row, CASE_LOWER);

        sqlsrv_fetch($stmt2);

        $index = 0;
        foreach ($row as $fieldname => $fieldvalue) {
            if (!empty($fieldspec[$fieldname]) AND preg_match('/^(string)$/i', $fieldspec[$fieldname]['type'])) {
                $data = @sqlsrv_get_field($stmt2,
                                          $index,
                                          SQLSRV_PHPTYPE_STRING(SQLSRV_ENC_BINARY));
                if ($data !== false) {
                    $data = iconv("utf-16le", "utf-8", $data);
                    $row[$fieldname] = $data;
                } // if
            } else {
                $data = sqlsrv_get_field($stmt2, $index);
            } // if
            $index++;
        } // foreach

        return $row;

    } // _convert_from_utf16_to_utf8

    // ****************************************************************************
    function getData_serial ($dbname, $tablename, $where, $rdc_limit=null, $rdc_offset=null)
    // Get data from a database table using optional 'where' criteria.
    // Return $result, not an array of data, so that individual rows can
    // be retrieved using the fetchRow() method.
    {
        if (is_null($where)) {
            $where = '';  // set to empty string (for version 8.1)
        } // if

        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        $this->connect($dbname) or trigger_error('SQLSRV', E_USER_ERROR);

        $pageno         = (int)$this->pageno;
        $rows_per_page  = (int)$this->rows_per_page;
        if ($pageno < 1) {
            $pageno = 1; // default to first page
        } // if
        $this->numrows  = 0;
        $this->lastpage = 0;

        if (!empty($this->temporary_table)) {
            if (empty($this->temp_tables)) {
                $this->temp_tables[$tablename] = $tablename;
            } // if
            if ($tablename = $this->temporary_table) {
                $tablename = '#'.$tablename;  // must be prefixed with '#'
            } // if
        } // if

        // look for optional SELECT parameters, or default to all fields
        if (empty($this->sql_select)) {
            // the default is all fields
            $select_str = '*';
        } else {
            $select_str = $this->adjustSelect($this->sql_select);
        } // if

        // use specified FROM parameters, or default to current table name
        if (empty($this->sql_from)) {
            // the default is current table
            $from_str = $tablename;
        } else {
            $from_str = $this->sql_from;
            // insert <newline> in front of every JOIN statement for readability
            $search_array  = array("/(?<!\n)LEFT JOIN/", "/(?<!\n)RIGHT JOIN/", "/(?<!\n)CROSS JOIN/", "/(?<!\n)INNER JOIN/");
            $replace_array = array(      "\nLEFT JOIN",        "\nRIGHT JOIN",        "\nCROSS JOIN",        "\nINNER JOIN");
            $from_str = preg_replace($search_array, $replace_array, $from_str);
            $from_str = $this->adjustFrom($from_str, $where);
        } // if

        // incorporate optional 'where' criteria
        $where = trim($where);
        if (empty($where)) {
            $where_str = '';
        } else {
            $where_str = "\nWHERE " .$this->adjustWhere($where);
        } // if

        if (!empty($this->sql_where_append)) {
            if (empty($where_str)) {
                $where_str = "\nWHERE {$this->sql_where_append}";
            } else {
                $where_str .= " AND {$this->sql_where_append}";
            } // if
        } // if

        // incorporate optional GROUP BY parameters
        if (!empty($this->sql_groupby)) {
            $group_str = $this->adjustGroupBy ($select_str, $this->sql_groupby, $this->sql_orderby);
            $group_str = "\nGROUP BY " .$group_str;
        } else {
            $group_str = NULL;
        } // if

        $lock_str = null;
        //$lock_str = 'NOLOCK';
        if ($GLOBALS['transaction_has_started'] === TRUE) {
            // for SQL SERVER all serial reads are done with a separate connection, and the following
            // statement MUST be issued if a transaction has already started with another connection
            $this->query = "SET TRANSACTION ISOLATION LEVEL READ UNCOMMITTED";
            $result = sqlsrv_query($this->dbconnect, $this->query) or trigger_error('SQLSRV', E_USER_ERROR);
            // write query to log file, if option is turned on
            logSqlQuery ($dbname, null, $this->query);
            $this->query = '';
        } // if

        if (!empty($this->sql_derived_table)) {
            //$this->sql_derived_orderby = unqualifyOrderBy($this->sql_orderby) ." $this->sql_orderby_seq";
            $this->sql_derived_orderby = null;
            list($select_str, $from_str, $append_str) = $this->adjust_derived_query($select_str, $from_str, $this->sql_having, $this->sql_search, $tablename);
            $having_str        = null;
            $this->sql_orderby = unqualifyOrderBy($this->sql_orderby);
        } elseif (!empty($this->sql_having)) {
            list($select_str, $from_str, $where_str, $group_str, $having_str, $this->sql_orderby) = $this->adjustHaving ($select_str, $from_str, $where_str, $group_str, $this->sql_having, $this->sql_orderby, $this->sql_union);
            $group_str       = null;
            $having_str      = null;
            $this->sql_union = null;
            $append_str      = null;
        } else {
            $having_str = NULL;
            $append_str = NULL;
        } // if

        // incorporate optional sort order
        if (!empty($this->sql_derived_orderby)) {
            $sort_str          = null;
        } elseif (!empty($this->sql_orderby)) {
            if (preg_match('/^(NULL)$/i', $this->sql_orderby)) {
                $sort_str = null;  // convert literal 'NULL' to actual NULL
            } else {
                $this->sql_orderby = $this->adjustOrderBy($this->sql_orderby, $select_str, $from_str, $group_str, $this->sql_derived_table);
                if (!empty($this->sql_orderby)) {
                    $pattern = '/( asc| ascending| desc| descending)$/i';
                    if (preg_match($pattern, $this->sql_orderby, $regs)) {
                        $sort_str = $this->sql_orderby;  // already contains 'asc' or 'desc'
                    } else {
                        $sort_str = "$this->sql_orderby $this->sql_orderby_seq";
                    } // if
                } else {
                    $sort_str = null;
                } // if
            } // if
        } else {
            $sort_str = implode(',', $this->primary_key);
        } // if

        if (!is_null($rdc_limit) OR !is_null($rdc_offset)) {
        	$min_rows = $rdc_offset;
            if ($rdc_limit > 0) {
                $max_rows = $min_rows + $rdc_limit;
            } // if
        } elseif ($rows_per_page > 0) {
            //$limit_str = 'LIMIT ' .$rows_per_page .' OFFSET ' .($pageno - 1) * $rows_per_page;
            $min_rows = (($pageno - 1) * $rows_per_page) +1;
            $max_rows = ($min_rows + $rows_per_page) -1;
        } // if

        if (!empty($this->sql_CTE_name)) {
            if (!empty($this->sql_search)) {
                if (empty($where_str)) {
                    $search_str = "\nWHERE ".$this->sql_search;
                } else {
                    $search_str = " AND ".$this->sql_search;
                } // if
            } else {
                $search_str = null;
            } // if
            //if (!empty($this->sql_CTE_select)) {
            //    $this->sql_CTE_select = $this->fixReservedWords($this->sql_CTE_select);
            //} // if
            foreach ($this->sql_CTE_name as $ix => $cte_name) {
                $this->sql_CTE_name[$ix]   = $this->fixReservedWords($this->sql_CTE_name[$ix]);
                $this->sql_CTE_anchor[$ix] = $this->fixReservedWords($this->sql_CTE_anchor[$ix]);
            } // foreach

            if (!empty($this->sql_CTE_recursive)) {
                // insert "UNION ALL" between the anchor and recursive segments
                $this->sql_CTE_recursive = "\n  UNION ALL\n\n".$this->sql_CTE_recursive;
            } // if
        } // if

        if (!empty($having_str)) {
            $having_str = "\nHAVING $this->sql_having";
        } // if

        // build the query string and run it
        if (isset($min_rows) OR isset($max_rows)) {
            // read a selected number of records
            if (isset($min_rows) AND isset($max_rows)) {
                $rownum = "rownum BETWEEN $min_rows AND $max_rows";
            } else {
                $rownum = "rownum > $min_rows";
            } // if

            if (!empty($this->sql_CTE_name)) {
                if (!empty($sort_str)) {
                    $sort_str = "ORDER BY $sort_str";
                } // if
                $this->query = "WITH {$this->sql_CTE_name[0]}";

$pattern1 = <<< END_OF_REGEX
/
ROW_NUMBER\(\)[ ]OVER[ ]\(ORDER[ ]BY[ ]
(?P<sort_str>  # pattern name
.*
)              # end pattern
\)
/imsx
END_OF_REGEX;

                foreach ($this->sql_CTE_anchor as $ix => $anchor) {
                    if ($ix > 0) {
                        $this->query .= "), {$this->sql_CTE_name[$ix]}";
                    } // if
                    $this->query .= " {$this->sql_CTE_anchor[$ix]}";
                    if (empty($sort_str)) {
                        // extract any ORDER BY string from the current
                        if (preg_match($pattern1, $this->sql_CTE_anchor[$ix], $regs)) {
                            $cte_sort_str = 'ORDER BY '.$regs['sort_str'];
                        } // if
                    } // if
                    if ($ix == 0 AND !empty($this->sql_CTE_recursive)) {
                        $this->query .= "{$this->sql_CTE_recursive}";
                    } // if
                } // foreach
                if (empty($sort_str) AND !empty($cte_sort_str)) {
                    $sort_str = $cte_sort_str;  // replace with last known ORDER BY string
                } // if

                $this->query .= ")
-- end of CTE --
SELECT * FROM (
  SELECT $select_str
  , ROW_NUMBER() OVER ($sort_str) AS rownum
  FROM $from_str $search_str
) AS x WHERE $rownum";

            } elseif (!empty($this->sql_union)) {
                if (!empty($sort_str)) {
                    $sort_str = unqualifyOrderBy($sort_str);
                    $sort_str = "\nORDER BY $sort_str";
                } // if
                $this->query = "SELECT * FROM ("
                             . "\n  SELECT *, ROW_NUMBER() OVER ($sort_str) AS rownum "
                             . "\n  FROM ("
                             . "\n    SELECT $select_str"
                             . "\n    FROM $from_str $lock_str"
                             . "\n    $where_str $group_str $having_str"
                             . "\n    UNION ALL"
                             . "\n    {$this->sql_union}"
                             . "\n  ) AS x"
                             . "\n) AS y"
                             . "\nWHERE $rownum $sort_str";
                $this->sql_union = null;
            } else {
                if (!empty($sort_str)) {
                    $sort_str = "ORDER BY $sort_str";
                } // if
                $this->query = "SELECT * FROM ("
                             . "\n  SELECT $select_str\n, ROW_NUMBER() OVER ($sort_str) AS rownum"
                             . "\n  FROM $from_str $lock_str "
                             . "\n  $where_str $group_str $having_str $append_str"
                             . "\n) AS x WHERE $rownum";
            } // if
            $start_time = getMicroTime();
            $result = sqlsrv_query($this->dbconnect, $this->query, array(), array('scrollable' => SQLSRV_CURSOR_STATIC)) or trigger_error('SQLSRV', E_USER_ERROR);
            $end_time = getMicroTime();
        } else {
            // read all available records
            if (!empty($this->sql_CTE_name)) {
                if (!empty($sort_str)) {
                    $sort_str = "\nORDER BY $sort_str";
                } // if
                $this->query = "WITH {$this->sql_CTE_name[0]}";

                foreach ($this->sql_CTE_anchor as $ix => $anchor) {
                    if ($ix > 0) {
                        $this->query .= "), {$this->sql_CTE_name[$ix]}";
                    } // if
                    $this->query .= " {$this->sql_CTE_anchor[$ix]}";
                    if ($ix == 0 AND !empty($this->sql_CTE_recursive)) {
                        $this->query .= "{$this->sql_CTE_recursive}";
                    } // if
                } // foreach

                $this->query .= ")
-- end of CTE --
SELECT $select_str
  FROM $from_str $lock_str \n$where_str $group_str $having_str $sort_str";

            } elseif (!empty($this->sql_union)) {
                if (!empty($sort_str)) {
                    $sort_str = "\nORDER BY $sort_str";
                } // if
                $this->query = "SELECT $select_str "
                             . "\nFROM $from_str $lock_str "
                             . "\n$where_str $group_str $having_str"
                             . "\nUNION ALL"
                             . "\n{$this->sql_union}"
                             . "\n$sort_str";
                $this->sql_union = null;
            } else {
                if (!empty($group_str)) {
                    $sort_str = null;
                } // if
                if (!empty($append_str)) {
                    // move $sort_str to AFTER $append_str
                    $append_str .= "\nORDER BY ".unqualifyOrderBy($sort_str);
                    $sort_str    = null;
                } elseif (!empty($sort_str)) {
                    $sort_str = "\nORDER BY $sort_str";
                } // if
                $this->query = "SELECT $select_str"
                              ."\nFROM $from_str $lock_str"
                              ."\n$where_str $group_str $having_str $sort_str $append_str";
            } // if
            $start_time = getMicroTime();
            $result = sqlsrv_query($this->dbconnect, $this->query, array(), array('scrollable' => SQLSRV_CURSOR_STATIC)) or trigger_error('SQLSRV', E_USER_ERROR);
            $end_time = getMicroTime();
        } // if

        $this->numrows = sqlsrv_num_rows($result);

        if ($this->numrows == -1) {
            $this->numrows = 1;  // why does it return -1 ?????
        } // if

        // write query to log file, if option is turned on
        logSqlQuery ($dbname, $tablename, $this->query, $this->numrows, $start_time, $end_time);

        return $result;

    } // getData_serial

    // ****************************************************************************
    function getErrors ()
    {
        return $this->errors;

    } // getErrors

    // ****************************************************************************
    function getErrorNo ()
    // return number of last error.
    {
        $errno = null;

        if ($errors = sqlsrv_errors()) {
            $errno  = $errors[0]['code'];
        } // if

        return $errno;

    } // getErrorNo

    // ****************************************************************************
    function getErrorString ()
    // return string containing details of last error.
    {
        $string = '';

        if (!empty($this->error_string)) {
            $string = $this->error_string;
            $this->error_string = null;

		} elseif ($errors = sqlsrv_errors()) {
            if (is_array($errors)) {
            	foreach ($errors as $error) {
                    if (empty($string)) {
                    	$string  = $error['message'];
                    } else {
                        $string .= "<br>\n" .$error['message'];
                    } // if
                } // foreach
            } // if

        } else {
            //$conerr = getLanguageText('sys0001', $this->dbname); // 'Cannot connect to database'
            $conerr = $GLOBALS['php_errormsg'];
            $string = $conerr;
        } // if

        return $string;

    } // getErrorString

    // ****************************************************************************
    function getErrorString2 ()
    // return additional information.
    {
        if ($this->dbconnect) {
        	$string  = 'Client Info: ' .$this->client_info ."<br>\n";
            $string .= 'Server Info: ';
            $server_info = sqlsrv_server_info($this->dbconnect);
            foreach ($server_info as $key => $value) {
                $string .= "$key: $value, ";
            } // if
            $string = rtrim($string, ', ');
        } else {
            $string = '';
        } // if

        return $string;

    } // getErrorString2

    // ****************************************************************************
    function getLastPage ()
    // return the last page number for retrieved rows.
    {
        return (int)$this->lastpage;

    } // getLastPage

    // ****************************************************************************
    function getNumRows ()
    // return the number of rows retrived for the current page.
    {
        return (int)$this->numrows;

    } // getNumRows

    // ****************************************************************************
    function getPageNo ()
    // get current page number to be retrieved for a multi-page display
    {
        if (empty($this->pageno)) {
            return 0;
        } else {
            return (int)$this->pageno;
        } // if

    } // getPageNo

    // ****************************************************************************
    function getQuery ()
    // return the last query string that was used
    {
        return $this->query;

    } // getQuery

    // ****************************************************************************
    function insertRecord ($dbname, $tablename, $fieldarray)
    // insert a record using the contents of $fieldarray.
    {
        $this->errors = array();

        $this->numrows = 0;  // record not inserted (yet)

        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        $this->connect($dbname) or trigger_error('SQLSRV', E_USER_ERROR);

        // get field specifications for this database table
        $fieldspec = $this->fieldspec;

        if (!preg_match('/^(audit)$/i', $dbname) AND isset($GLOBALS['mode']) AND preg_match('/^(blockchain)$/i', $GLOBALS['mode'])) {
            // update transferred by blockchain, so don't overwrite these values
        } else {
            foreach ($fieldspec as $field => $spec) {
                if (empty($fieldarray[$field]) OR (!empty($spec['default']) AND $fieldarray[$field] == $spec['default'])) {
                    // look for fields with 'autoinsert' option set
                    if (array_key_exists('autoinsert', $spec)) {
    				    switch ($spec['type']){
    					    case 'datetime':
                            case 'timestamp':
    						    $fieldarray[$field] = getTimeStamp(null, true);
    						    break;
    					    case 'date':
    						    $fieldarray[$field] = getTimeStamp('date', true);
    						    break;
    					    case 'time':
    						    $fieldarray[$field] = getTimeStamp('time', true);
    					        break;
    					    case 'string':
    						    $fieldarray[$field] = $_SESSION['logon_user_id'];
    						    break;
    					    default:
    						    // do nothing
    				    } // switch
                    } // if
                } // if
            } // foreach
        } // if

        // find out if any field in the primary key has 'identity' (auto_increment) set
		$auto_increment  = '';
        $identity_insert = FALSE;
		foreach ($this->primary_key as $pkey){
			if (isset($fieldspec[$pkey]['auto_increment'])) {
			    $this->retry_on_duplicate_key = null;  // this feature cannot be used with auto_increment
			    if (!empty($fieldarray[$pkey]) AND $fieldarray[$pkey] > 0) {
			    	// value has been supplied manually, so set this to stop an error
                    $this->query = "SET IDENTITY_INSERT [{$tablename}] ON";
                    $result = sqlsrv_query($this->dbconnect, $this->query) or trigger_error('SQLSRV', E_USER_ERROR);
                    logSqlQuery ($dbname, null, $this->query);
                    $identity_insert = TRUE;
			    } else {
    			    $auto_increment = $pkey;                // save name of related sequence
    				unset($fieldarray[$auto_increment]);    // remove from data array
			    } // if
			} // if
		} // foreach

		if (!empty($this->retry_on_duplicate_key)) {
        	if (!array_key_exists($this->retry_on_duplicate_key, $fieldspec)) {
        	    // this field does not exist, so remove it
        		$this->retry_on_duplicate_key = null;
        	} // if
        } // if

        // build 'where' string using values for primary key
	    $primary_key = $this->buildKeyString ($fieldarray, $this->primary_key);

        if (empty($auto_increment) AND empty($this->retry_on_duplicate_key)) {
	        // find out if a record with this primary key already exists
	        $query = "SELECT count(*) FROM $tablename WHERE $primary_key";
	        $count = $this->getCount($dbname, $tablename, $query);
	        // Is this primary key taken?
	        if ($count <> 0) {
	            if (is_True($this->no_duplicate_error)) {
	                // exit without setting an error
	                return $fieldarray;

	            } elseif (is_True($this->update_on_duplicate_key)) {
                    // switch to 'update'
                    $old_array = where2array($primary_key);
                    $fieldarray = $this->updateRecord ($dbname, $tablename, $fieldarray, $old_array);
                    return $fieldarray;

                } else {
	            	// set error message for each field within this key
    	            foreach ($this->primary_key as $fieldname) {
    	                $this->errors[$fieldname] = getLanguageText('sys0002'); // 'A record already exists with this ID.'
    	            } // foreach
    	            $this->query = $query;  // save this in case trigger_error() is called
	            } // if
	            return $fieldarray;
	        } // if
		} // if

        // validate any optional unique/candidate keys
        if (!empty($this->unique_keys)) {
            // there may be several keys with several fields in each
            foreach ($this->unique_keys as $key) {
                $where = $this->buildKeyString ($fieldarray, $key);
                if (!empty($where)) {
                    $query = "SELECT count(*) FROM $tablename WHERE $where";
                    $count = $this->getCount($dbname, $tablename, $query);
                    if ($count <> 0) {
                        if (is_True($this->no_duplicate_error)) {
    	                    // exit without setting an error
    	                    return $fieldarray;
    	                } else {
                            // set error message for each field within this key
                            reset($key);
                            foreach ($key as $fieldname) {
                                $this->errors[$fieldname] = getLanguageText('sys0003'); // 'A record already exists with this key.'
                            } // foreach
                            $this->errors['where'] = $where;
                            $this->query = $query;  // save this in case trigger_error() is called
                            return $fieldarray;
    	                } // if
                    } // if
                } // if
            } // foreach
        } // if

        $repeat       = false;
        $repeat_count = 0;
        $pattern1 = '/(integer|decimal|numeric|float|real|double)/i';
        $pattern2 = '/^\w+[ ]*\(.*\)$/imsx';  // function(...)
        $start_time = getMicroTime();
        do {
            // insert this record into the database
            $cols = '';
            $vals = '';
            foreach ($fieldarray as $item => $value) {
                if (is_null($value)) $value = '';  // set to empty string (for version 8.1)
                if (preg_match('/set|array|varray/i', $fieldspec[$item]['type'])) {
                    if (!empty($value)) {
                    	// assume a one-dimensional array
                    	$array1  = explode(',', $value);
                    	$string1 = '';
                    	foreach ($array1 as $value1) {
                    		if (empty($string1)) {
                    			$string1 = '"' .$value1 .'"';
                    		} else {
                    		    $string1 .= ', "' .$value1 .'"';
                    		} // if
                    	} // foreach
                    	// enclose array in curly braces
                    	$value .= "[$item]='{" .$string1 ."}', ";
                    } // if
                } // if
                if (!array_key_exists('required',$fieldspec[$item])
                AND strlen($value) == 0 OR strtoupper(trim($value)) == 'NULL') {
                    // null entries are set to NULL, not '' (there is a difference!)
                    $cols .= "[$item], ";
                    $vals .= "NULL, ";
                } elseif (strlen($value) == 0 AND isset($fieldspec[$item]['type']) AND preg_match($pattern1, $fieldspec[$item]['type'])) {
                    // cannot use empty string for numeric fields, so use NULL instead
                    $cols .= "[$item], ";
                    $vals .= "NULL, ";
                } elseif ($fieldspec[$item]['type'] == 'blob' AND substr($value, 0, 2) == '0x') {
                    // this is a headecimal number, so don't enclose in quotes
                    $cols .= "[$item], ";
                    $vals .= "$value, ";
                //} elseif ($fieldspec[$item]['type'] == 'blob') {
                //    // convert any dodgy characters to binary
                //    $cols .= "[$item], ";
                //    $value = $this->adjustData($value);
                //    $vals .= "CAST('$value' AS binary), ";
                } elseif (is_array($this->allow_db_function) AND in_array($item, $this->allow_db_function) AND preg_match($pattern2, $value)) {
                    // this is a function, so change to the new value (without enclosing quotes)
                    $cols .= "[$item], ";
                    $vals .= $this->adjustData($value) .", ";
                } elseif ($fieldspec[$item]['type'] == 'string') {
                    // prefix with 'N' to force acceptance of UTF-8 characters
                    $cols .= "[$item], ";
                    $vals .= "N'" .$this->adjustData($value) ."', ";
                } else {
                    $cols .= "[$item], ";
                    $vals .= "'" .$this->adjustData($value) ."', ";
                } // if
            } // foreach

            // remove trailing commas
            $cols = rtrim($cols, ', ');
            $vals = rtrim($vals, ', ');

            $this->query = 'INSERT INTO ' .$tablename .' (' .$cols .') VALUES (' .$vals .')';
            //$result = sqlsrv_query($this->dbconnect, $this->query) or trigger_error('SQLSRV', E_USER_ERROR);
            $result = sqlsrv_query($this->dbconnect, $this->query);
            if ($result === false) {
                $errno = $this->getErrorNo();
                if ($errno == 2627) {
                    if (!empty($this->retry_on_duplicate_key)) {
                        // increment the specified field and try again
                        $spec = $fieldspec[$this->retry_on_duplicate_key];
                        if (isset($spec['precision']) AND $spec['precision'] == 38) {
                            $sum = gmp_add($fieldarray[$this->retry_on_duplicate_key], 1);
                            $fieldarray[$this->retry_on_duplicate_key] = gmp_strval($sum);
                        } else {
                            $fieldarray[$this->retry_on_duplicate_key]++;
                        } // if
                        $repeat = true;
                        $repeat_count++;
                        if ($repeat_count > 100) {
                            // too many retries, so turn this feature off
                    	    $this->retry_on_duplicate_key = null;
                        } // if

                    } elseif (is_True($this->update_on_duplicate_key)) {
                        // switch to 'update'
                        $old_array = where2array($primary_key);
                        $fieldarray = $this->updateRecord ($dbname, $tablename, $fieldarray, $old_array);
                        return $fieldarray;

                    } elseif (is_True($this->no_duplicate_error)) {
                        // this is a duplicate, but don't fail
                        $this->numrows = 0;

                    } else {
                        trigger_error('SQLSRV', E_USER_ERROR);
                    } // if

                } else {
                    trigger_error('SQLSRV', E_USER_ERROR);
                } // if
            } else {
                $repeat = false;
                $this->numrows = 1;  // record has been inserted
            } // if
        } while ($repeat == true);
        $end_time = getMicroTime();

        // write query to log file, if option is turned on
        logSqlQuery ($dbname, $tablename, $this->query, 1, $start_time, $end_time);

		if (!empty($auto_increment)) {
			// obtain the last value used by auto_increment
			//$this->query = "SELECT @@identity";
            $this->query = "SELECT SCOPE_IDENTITY()";
            $start_time = getMicroTime();
            $result = sqlsrv_query($this->dbconnect, $this->query) or trigger_error('SQLSRV', E_USER_ERROR);
            $identity = sqlsrv_fetch_array($result, SQLSRV_FETCH_NUMERIC);
            $end_time = getMicroTime();
            $fieldarray[$auto_increment] = $identity[0];
            $primary_key = $this->buildKeyString ($fieldarray, $this->primary_key);
            // write query to log file, if option is turned on
            logSqlQuery ($dbname, $tablename, $this->query, $fieldarray[$auto_increment], $start_time, $end_time);

        } elseif ($identity_insert == TRUE) {
            $this->query = "SET IDENTITY_INSERT [{$tablename}] OFF";
            $result = sqlsrv_query($this->dbconnect, $this->query) or trigger_error('SQLSRV', E_USER_ERROR);
            logSqlQuery ($dbname, null, $this->query);
            $identity_insert = FALSE;
		} // if

        if ($this->audit_logging) {
        	if (defined('TRANSIX_NO_AUDIT') OR defined('NO_AUDIT_LOGGING')) {
        		// do nothing
	        } else {
	            $auditobj = RDCsingleton::getInstance('audit_tbl');
	            // add record details to audit database
	            $auditobj->auditInsert($dbname, $tablename, $this->fieldspec, $primary_key, $fieldarray);
	            $this->errors = array_merge($auditobj->getErrors(), $this->errors);
			} // if
        } // if

        $this->numrows = 1;  // record has been inserted

        return $fieldarray;

    } // insertRecord

    // ****************************************************************************
    function multiQuery ($dbname, $tablename, $query_array)
    // perform multiple queries in a single step
    {
        $this->errors = array();

        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        $this->connect($dbname) or trigger_error('SQLSRV', E_USER_ERROR);

        //static $temp_tables;
        //if (!is_array($temp_tables)) {
        //    $temp_tables = array();
        //} // if
        //$this->temp_tables =& $temp_tables;

        $single_step = true;  // special code to debug each query separately
        if ($single_step) {
            // execute these statements one at a time
            foreach ($query_array as $query_part) {
                $start_time = getMicroTime();

                if (preg_match('/\bif[ ]*\(/i', $query_part)) {
                    // convert 'IF(...)' to 'IIF(...)')
                    $query_part = preg_replace('/\bif[ ]*\(/i', 'IIF(', $query_part);
                } // if

                $this->query = $query_part;
                $skip_errors = array();  // list of allowable errors
                $this->query = $this->adjustTempTable($this->query, $this->temp_tables, $skip_errors);
                $this->query = $this->fixReservedWords($this->query);
                $result = sqlsrv_query($this->dbconnect, $this->query);
                if ($result === false) {
                    $errno  = $this->getErrorNo();
                    if (in_array($errno, $skip_errors)) {
                        // ignore this error
                    } else {
                        trigger_error('SQLSRV', E_USER_ERROR);
                    } // if
                } // if

                $end_time = getMicroTime();
                $this->numrows = sqlsrv_rows_affected ($result);
                // write query to log file, if option is turned on
                logSqlQuery ($dbname, $tablename, $this->query, $this->numrows, $start_time, $end_time);
            } // foreach
            if (count($query_array) == 1) {
                // only a single query, so return result in full
                $this->numrows = sqlsrv_rows_affected ($result);
                if (is_bool($result)) {
                    return $result;
                } elseif (preg_match('/^(INSERT|UPDATE|DELETE)/i', $this->query)) {
                    return sqlsrv_rows_affected ($result);
                } else {
                    $array = array();
                    while ($row = sqlsrv_fetch_array($result, SQLSRV_FETCH_ASSOC)) {
                        $array[] = array_change_key_case($row, CASE_LOWER);
                    } // while
                    sqlsrv_free_stmt($result);
                    if (preg_match('/^(SELECT COUNT\()/i', $this->query)) {
                        return array_shift($array[0]);
                    } // if
                    return $array;
                } // if
            } // if
            return true;
        } // if

        return $result;

    } // multiQuery

    // ****************************************************************************
    function prepareAndExecuteQuery ($dbname, $tablename, $query, $params, $output_type='string')
    // prepare and execute a single query, with an optional result
    {
        $this->errors = array();

        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        $this->connect($dbname) or trigger_error('SQLSRV', E_USER_ERROR);

        $stmnt = sqlsrv_query($this->dbconnect, $query, $params) or trigger_error('SQLSRV', E_USER_ERROR);

        if (!empty($output_type)) {
            while ( sqlsrv_fetch( $stmnt)) {
                switch ($output_type) {
                    case 'string':
                        $result = sqlsrv_get_field( $stmnt, 0, SQLSRV_PHPTYPE_STRING ( SQLSRV_ENC_CHAR)) or trigger_error('SQLSRV', E_USER_ERROR);
                        break;
                    default:
                        $result = sqlsrv_get_field( $stmnt, 0) or trigger_error('SQLSRV', E_USER_ERROR);
                } // switch
            } // while
        } // if

        sqlsrv_free_stmt($stmnt);

        return $result;

    } // prepareAndExecuteQuery

    // ****************************************************************************
    function rollback ($dbname)
    // rollback this transaction due to some sort of error.
    {
        $this->errors = array();

        if (!$this->dbconnect) {
            // not connected yet, so do nothing
            return FALSE;
        } // if

        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        $start_time = getMicroTime();
        $result = sqlsrv_rollback($this->dbconnect);
        $end_time = getMicroTime();

        // write query to log file, if option is turned on
        logSqlQuery ($dbname, null, 'ROLLBACK', null, $start_time, $end_time);
        $this->query = '';

        if (defined('TRANSIX_NO_AUDIT') OR defined('NO_AUDIT_LOGGING')) {
        	// do nothing
	    } else {
            $auditobj = RDCsingleton::getInstance('audit_tbl');
            $result = $auditobj->close();
        } // if

        return $result;

    } // rollback

    // ****************************************************************************
    function selectDB ($dbname)
    // select a different database via the current connection.
    {
        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        $this->query = "USE [$dbname]";
        $result = sqlsrv_query($this->dbconnect, $this->query) or trigger_error('SQLSRV', E_USER_ERROR);

        // write query to log file, if option is turned on
        logSqlQuery ($dbname, null, $this->query);
        $this->query = '';

        return true;

    } // selectDB

    // ****************************************************************************
    function setErrorString ($string)
    // capture string from last non-fatal error.
    {
        $this->error_string = trim($string);

        return;

    } // setErrorString

    // ****************************************************************************
    function setOrderBy ($sql_orderby)
    // this allows a sort order to be specified (see getData)
    {
        $this->sql_orderby = trim($sql_orderby);

    } // setOrderBy

    // ****************************************************************************
    function setOrderBySeq ($sql_orderby_seq)
    // this allows a sort sequence ('asc' or 'desc') to be set (see getData)
    {
        $this->sql_orderby_seq = trim($sql_orderby_seq);

    } // setOrderBySeq

    // ****************************************************************************
    function setPageNo ($pageno='1')
    // this allows a particular page number to be selected (see getData)
    {
        $this->pageno = (int)$pageno;

    } // setPageNo

    // ****************************************************************************
    function setRowLocks ($level=null, $supplemental=null)
    // set row-level locks on next SELECT statement
    {
        // upshift first two characters
        $level = substr(strtoupper((string)$level),0,2);

        switch ($level){
            case 'SH':
                $this->row_locks = 'SH';
                break;
            case 'EX':
                $this->row_locks = 'EX';
                break;
            default:
                $this->row_locks = null;
        } // switch

        $this->row_locks_supp = $supplemental;

        return;

    } // setRowLocks

    // ****************************************************************************
    function setRowsPerPage ($rows_per_page)
    // this allows the default value to be changed
    {
        if ($rows_per_page > 0) {
            $this->rows_per_page = (int)$rows_per_page;
        } // if

    } // setRowsPerPage

    // ****************************************************************************
    function setSqlSearch ($sql_search)
    // set additional criteria to be used in sql select
    {
        $this->sql_search = trim($sql_search);

    } // setSqlSearch

    // ****************************************************************************
    function modifyOrderBy ($sort_str, $select_str, $sql_select)
    // modify the ORDER BY string to work in an outer query
    // $sort_str may be modified
    // $select_str = may be ether '*' or the same as $sql_select
    // $sql_select = the original SELECT string
    {
        $sort_str_out = null;

        $pattern1 = <<< END_OF_REGEX
/
^               # begins with
(?<table>\w+)   # <tablename>
\.              # dot
(?<column>\w+   # <columnname>
(\s+(ascending|asc|descending|desc))?   # 'ASC|DESC' (optional)
)
$               # ends with
/imsx
END_OF_REGEX;

        $pattern2 = <<< END_OF_REGEX
/
^
(?<expression>.+?)
(?<direction>(ascending|asc|descending|desc))?   # 'ASC|DESC' (optional)
$
/xims
END_OF_REGEX;

        // get list of '(expression)=$alias' entries in the SELECT string
        $alias_names = extractAliasNames($sql_select);
        $alias_names = array_flip($alias_names);
        $alias_names = array_change_key_case($alias_names);

        $sort_array = extractOrderBy($sort_str);
        $sort_array = array_map('trim', $sort_array);

        // examine contents of $sort_array
        foreach ($sort_array as $ix => $fieldname) {
            $direction = null;
            if (preg_match($pattern1, $fieldname, $regs1)) {
                $sort_array[$ix] = $fieldname;
            } elseif (preg_match($pattern2, $fieldname, $regs2)) {
                $expression = trim($regs2['expression']);
                if (!empty($regs2['direction'])) {
                    $direction  = trim($regs2['direction']);
                } // if
                if ($select_str == '*') {
                    // $select_str does not contain any expressions, so ...
                    if (array_key_exists($expression, $alias_names)) {
                        // swap the expression for its alias name
                        if (!empty($direction)) {
                            $sort_array[$ix] = $alias_names[$expression]." $direction";
                        } else {
                            $sort_array[$ix] = $alias_names[$expression];
                        } // if
                    } // if
                } // if
            } // if
        } // foreach

        $sort_str_out = implode(', ', $sort_array);

        return $sort_str_out;

    } // modifyOrderBy

    // ****************************************************************************
    function modifyOrderByDerived ($sort_str, $inner_select, $outer_select, $derived_table)
    // modify the ORDER BY string to work in a query with a derived table
    // $sort_str may be modified
    // $inner_select = the select string for the derived query
    // $outer_select = the select string for the outer query
    {
        $sort_str_out = null;

        $pattern1 = <<< END_OF_REGEX
/
^               # begins with
(?<table>\w+)   # <tablename>
\.              # dot
(?<column>\w+   # <columnname>
(\s+(ascending|asc|descending|desc))?   # 'ASC|DESC' (optional)
)
$               # ends with
/imsx
END_OF_REGEX;

        $pattern2 = <<< END_OF_REGEX
/
^
(?<expression>.+?)
(?<direction>(ascending|asc|descending|desc))?   # 'ASC|DESC' (optional)
$
/xims
END_OF_REGEX;

        // get list of '(expression)=$alias' entries in the SELECT string
        $inner_names = extractFieldNamesAssoc($inner_select);
        $inner_names = array_change_key_case($inner_names);
        $inner_names_unq = unqualifyFieldArray($inner_names);  // try without table names

        $outer_names = extractFieldNamesAssoc($outer_select);
        $outer_names = array_change_key_case($outer_names);
        $outer_names_unq = unqualifyFieldArray($outer_names);  // try without table names

        $sort_array = extractOrderBy($sort_str);
        $sort_array = array_map('trim', $sort_array);

        // examine contents of $sort_array
        foreach ($sort_array as $ix => $fieldname) {
            $direction = null;
            if (preg_match($pattern1, $fieldname, $regs1)) {
                $sort_array[$ix] = $fieldname;
            } elseif (preg_match($pattern2, $fieldname, $regs2)) {
                $expression = trim($regs2['expression']);
                if (!empty($regs2['direction'])) {
                    $direction  = trim($regs2['direction']);
                } // if
                if (array_key_exists($expression, $inner_names) OR array_key_exists($expression, $inner_names_unq)) {
                    // swap the expression for its alias name
                    if (!empty($direction)) {
                        $sort_array[$ix] = "$derived_table.$expression $direction";
                    } else {
                        $sort_array[$ix] = "$derived_table.$expression";
                    } // if

                } elseif (array_key_exists($expression, $outer_names)) {
                    if (!empty($direction)) {
                        $sort_array[$ix] = $outer_names[$expression].' '.$direction;
                    } else {
                        $sort_array[$ix] = $outer_names[$expression];
                    } // if
                } // if
            } // if
        } // foreach

        $sort_str_out = implode(', ', $sort_array);

        return $sort_str_out;

    } // modifyOrderByDerived

    // ****************************************************************************
    function startTransaction ($dbname)
    // start a new transaction, to be terminated by either COMMIT or ROLLBACK.
    {
        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        $this->connect($dbname) or trigger_error('SQLSRV', E_USER_ERROR);

        $start_time = getMicroTime();
        $result = sqlsrv_begin_transaction($this->dbconnect);
        if (!$result) {
            $errors = sqlsrv_errors();
            trigger_error($errors[0]['code'], E_USER_ERROR);
        } // if
        $end_time = getMicroTime();

        // write query to log file, if option is turned on
        logSqlQuery ($dbname, null, 'BEGIN TRANSACTION', null, $start_time, $end_time);
        $this->query = '';

        //if (!empty($this->table_locks)) {
        //	$result = $this->_setDatabaseLock($this->table_locks);
        //} // if

        return $result;

    } // startTrasaction

    // ****************************************************************************
    function updateRecord ($dbname, $tablename, $fieldarray, $oldarray, $where=null)
    // update a record using the contents of $fieldarray.
    {
        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        $this->connect($dbname) or trigger_error('SQLSRV', E_USER_ERROR);

        // get field specifications for this database table
        $fieldspec = $this->fieldspec;

        if (strlen($where) == 0) {
            // build 'where' string using values for primary key
            $where = $this->buildKeyString ($oldarray, $this->primary_key);
            if (empty($where)) {
                $where = $this->buildKeyString ($fieldarray, $this->primary_key);
                if (empty($where)) {
                    $where = 'PRIMARY_KEY_IS_MISSING';
                } // if
            } // if
        } else {
        	// use $where as supplied, and remove pkey specs so their values can be changed
            $this->unique_keys[] = $this->primary_key;  // but still check for duplicate value
        	$this->primary_key = array();
        } // if

        // validate any optional unique/candidate keys
        if (!empty($this->unique_keys)) {
            // there may be several keys with several fields in each
            foreach ($this->unique_keys as $key) {
                $where1 = $this->buildKeyString ($oldarray, $key);
                $where2 = $this->buildKeyString ($fieldarray, $key);
                //if (strlen($where2) > 0 AND $where1 <> $where2) {
                if (strlen($where2) > 0 AND strcasecmp($where1, $where2) <> 0) {
                    // key has changed, so check for uniqueness
                    $query = "SELECT count(*) FROM $tablename WHERE $where2";
                    $count = $this->getCount($dbname, $tablename, $query);
                    if ($count <> 0) {
                        // set error message for each field within this key
                        reset($key);
                        foreach ($key as $fieldname) {
                            $this->errors[$fieldname] = getLanguageText('sys0003'); // 'A record already exists with this key.'
                        } // foreach
                        $this->errors['where'] = $where2;
                        $this->query = $query;  // save this in case trigger_error() is called
                        return $fieldarray;
                    } // if
                } // if
            } // foreach
        } // if

        // remove any values that have not changed
        $fieldarray = getChanges($fieldarray, $oldarray);

        if (empty($fieldarray)) {
            // nothing to update, so return now
            $this->numrows = 0;
            return $fieldarray;
        } // if

        if (isset($GLOBALS['mode']) AND preg_match('/^(logon)$/', $GLOBALS['mode']) AND $tablename == 'mnu_user') {
            // do not set these fields when logging in
        } elseif (isset($GLOBALS['mode']) AND preg_match('/^(blockchain)$/i', $GLOBALS['mode'])) {
            // update transferred by blockchain, so don't overwrite these values
        } else {
            foreach ($fieldspec as $field => $spec) {
                // look for fields with 'autoupdate' option set
                if (array_key_exists('autoupdate', $spec)) {
                    switch ($spec['type']){
    					case 'datetime':
                            if (empty($fieldarray[$field])) {
    						    $fieldarray[$field] = getTimeStamp(null, true);
    					    } // if
    						break;
    					case 'date':
    					    if (empty($fieldarray[$field])) {
    						    $fieldarray[$field] = getTimeStamp('date', true);
    					    } // if
    						break;
    					case 'time':
    					    if (empty($fieldarray[$field])) {
						        $fieldarray[$field] = getTimeStamp('time', true);
    					    } // if
						    break;
					    case 'string':
					        if (empty($fieldarray[$field])) {
    						    $fieldarray[$field] = $_SESSION['logon_user_id'];
					        } // if
    						break;
                        case 'integer':
					        $fieldarray[$field] = $oldarray[$field] +1;
					        break;
    					default:
    						// do nothing
    				} // switch
                } // if
            } // foreach
        } // if

        // build update string from non-pkey fields
        $update = '';
        $pattern1 = '/(integer|decimal|numeric|float|real)/i';
        $pattern2 = '/^\w+[ ]*\(.*\)$/imsx';  // function(...)
        foreach ($fieldarray as $item => $value) {
            // use this item if it IS NOT part of primary key
            if (!in_array($item, $this->primary_key)) {
                if (is_null($value) OR strtoupper(trim($value)) == 'NULL') {
                    // null entries are set to NULL, not '' (there is a difference!)
                    $update .= "[$item]=NULL,";
                } elseif (preg_match('/set|array|varray/i', $fieldspec[$item]['type'])) {
                    if (!empty($value)) {
                        // assume a one-dimensional array
                        $array1  = explode(',', $value);
                        $string1 = '';
                    	foreach ($array1 as $value1) {
                    	    $value1 = $this->adjustData($value1);
                    		if (empty($string1)) {
                    			$string1 = '"' .$value1 .'"';
                            } else {
                    		    $string1 .= ', "' .$value1 .'"';
                            } // if
                    	} // foreach
                        // enclose array in curly braces
                    	$update .= "[$item]='{" .$string1 ."}', ";
                    } // if
                } elseif (preg_match($pattern1, $fieldspec[$item]['type'], $match)) {
                    // do not enclose numbers in quotes (this also allows 'value=value+1'
                    if (strlen($value) == 0) {
                    	$update .= "[$item]=NULL,";
                    } else {
                        $update .= "[$item]=$value,";
                    } // if
                } elseif ($fieldspec[$item]['type'] == 'blob' AND substr($value, 0, 2) == '0x') {
                    // this is a headecimal number, so don't enclose in quotes
                    $update .= "[$item]=$value";
                } elseif (is_array($this->allow_db_function) AND in_array($item, $this->allow_db_function) AND preg_match($pattern2, $value)) {
                    // this is a function, so change to the new value (without enclosing quotes)
                    $update .= "[$item]=" .$this->adjustData($value) .", ";
                } elseif ($fieldspec[$item]['type'] == 'string') {
                    // prefix with 'N' to force acceptance of UTF-8 characters
                    $update .= "[$item]=N'" .$this->adjustData($value) ."', ";
                } else {
                    // change to the new value
                    $update .= "[$item]='" .$this->adjustData($value) ."', ";
                } // if
            } // if
        } // foreach

        // strip trailing comma
        $update = rtrim($update, ', ');

        // append WHERE clause to SQL query
        $this->query = "UPDATE $tablename SET $update WHERE $where";
        $start_time = getMicroTime();
        $result = sqlsrv_query($this->dbconnect, $this->query) or trigger_error('SQLSRV', E_USER_ERROR);
        $end_time = getMicroTime();

        // get count of affected rows as there may be more than one
        $this->numrows = sqlsrv_rows_affected($result);

        // write query to log file, if option is turned on
        logSqlQuery ($dbname, $tablename, $this->query, $this->numrows, $start_time, $end_time);

        if ($this->audit_logging) {
        	if (defined('TRANSIX_NO_AUDIT') OR defined('NO_AUDIT_LOGGING')) {
        		// do nothing
	        } else {
	            $auditobj = RDCsingleton::getInstance('audit_tbl');
	            // add record details to audit database
	            $auditobj->auditUpdate($dbname, $tablename, $this->fieldspec, $where, $fieldarray, $oldarray);
	            $this->errors = array_merge($auditobj->getErrors(), $this->errors);
			} // if
        } // if

        return $fieldarray;

    } // updateRecord

    // ****************************************************************************
    function updateSelection ($dbname, $tablename, $replace, $selection)
    // update a selection of records in a single operation.
    {
        $this->errors = array();

        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        $this->connect($dbname) or trigger_error('SQLSRV', E_USER_ERROR);

        $this->query = "UPDATE $tablename SET $replace WHERE $selection";
        $start_time = getMicroTime();
        $result = sqlsrv_query($this->dbconnect, $this->query) or trigger_error('SQLSRV', E_USER_ERROR);
        $end_time = getMicroTime();

        $count = sqlsrv_rows_affected($result);

        // write query to log file, if option is turned on
        logSqlQuery ($dbname, $tablename, $this->query, $count, $start_time, $end_time);

        if ($count > 0) {
            if ($this->audit_logging AND !defined('TRANSIX_NO_AUDIT')) {
            	if (defined('TRANSIX_NO_AUDIT') OR defined('NO_AUDIT_LOGGING')) {
        			// do nothing
		        } else {
                    $auditobj = RDCsingleton::getInstance('audit_tbl');
	                // add record details to audit database
                    $auditobj->auditUpdateSelection($dbname, $tablename, $this->fieldspec, $selection, $replace);
	                $this->errors = array_merge($auditobj->getErrors(), $this->errors);
				} // if
            } // if
        } // if

        return $count;

    } // updateSelection

    // ****************************************************************************
    // the following are DDL (Data Definition Language) methods
    // ****************************************************************************
    function ddl_getColumnSpecs ()
    // return the array of column specifications.
    {

        $colspecs['bigint']     = array('name' => 'BIGINT',
                                        'type' => 'integer',
                                        'minvalue' => '-9223372036854775808',
                                        'maxvalue' => '9223372036854775807',
                                        'size' => 20);
        $colspecs['int']        = array('name' => 'INTEGER',
                                        'type' => 'integer',
                                        'minvalue' => -2147483648,
                                        'maxvalue' => 2147483647,
                                        'size' => 11);
        $colspecs['integer']    = array('name' => 'INTEGER',
                                        'type' => 'integer',
                                        'minvalue' => -2147483648,
                                        'maxvalue' => 2147483647,
                                        'size' => 11);
        $colspecs['smallint']   = array('name' => 'SMALLINT',
                                        'type' => 'integer',
                                        'minvalue' => -32768,
                                        'maxvalue' => 32767,
                                        'size' => 6);
        $colspecs['tinyint']    = array('name' => 'TINYINT',
                                        'type' => 'integer',
                                        'minvalue' => 0,
                                        'maxvalue' => 255,
                                        'size' => 3);
        $colspecs['decimal']    = array('name' => 'DECIMAL',
                                        'type' => 'numeric',
                                        'size' => 38);
        $colspecs['numeric']    = array('name' => 'NUMERIC',
                                        'type' => 'numeric',
                                        'size' => 38);

        $colspecs['money']      = array('name' => 'MONEY',
                                        'type' => 'numeric',
                                        'precision' => 19,
                                        'scale' => 4,
                                        'minvalue' => -922,337,203,685,477.5808,
                                        'maxvalue' => 922,337,203,685,477.5807,
                                        'size' => 20);
        $colspecs['smallmoney'] = array('name' => 'SMALLMONEY',
                                        'type' => 'numeric',
                                        'precision' => 10,
                                        'scale' => 4,
                                        'minvalue' => -214748.3648,
                                        'maxvalue' => 214748.3647,
                                        'size' => 11);
        $colspecs['bit']        = array('name' => 'BIT',
                                        'type' => 'bit',
                                        'size' => 1);
        $colspecs['float']      = array('name' => 'FLOAT',
                                        'type' => 'float',
                                        'size' => 53);
        $colspecs['real']       = array('name' => 'REAL',
                                        'type' => 'float',
                                        'size' => 24);
        $colspecs['date']           = array('name' => 'DATE',
                                            'type' => 'date',
                                            'size' => 12);
        $colspecs['time']           = array('name' => 'TIME',
                                            'type' => 'time',
                                            'size' => 8);

        $colspecs['smalldatetime']  = array('name' => 'DATETIME',
                                            'type' => 'datetime',
                                            'size' => 20);
        $colspecs['datetime']       = array('name' => 'DATETIME',
                                            'type' => 'datetime',
                                            'size' => 24);
        $colspecs['datetime2']      = array('name' => 'DATETIME',
                                            'type' => 'datetime',
                                            'size' => 28);
        $colspecs['datetimeoffset'] = array('name' => 'DATETIMEOFFSET',
                                            'type' => 'datetime',
                                            'size' => 35);

        $colspecs['char']       = array('name' => 'CHAR',
                                        'type' => 'string',
                                        'size' => 8000);
        $colspecs['varchar']    = array('name' => 'CHARACTER VARYING',
                                        'type' => 'string',
                                        'size' => 2147483647);
        $colspecs['text']       = array('name' => 'TEXT',
                                        'type' => 'string',
                                        'size' => 2147483647);
        $colspecs['nchar']      = array('name' => 'NATIONAL CHAR',
                                        'type' => 'string',
                                        'size' => 4000);
        $colspecs['nvarchar']   = array('name' => 'NATIONAL CHARACTER VARYING',
                                        'type' => 'string',
                                        'size' => 2147483647);
        $colspecs['ntext']      = array('name' => 'NATIONAL TEXT',
                                        'type' => 'string',
                                        'size' => 1073741823);
        $colspecs['image']      = array('name' => 'IMAGE',
                                        'type' => 'blob',
                                        'size' => 2147483647);
        $colspecs['binary']     = array('name' => 'IMAGE',
                                        'type' => 'blob',
                                        'size' => 8000);
        $colspecs['varbinary']  = array('name' => 'IMAGE',
                                        'type' => 'blob',
                                        'size' => 2147483647);

        $colspecs['timestamp']  = array('name' => 'ROWVERSION',
                                        'type' => 'rowversion');
        $colspecs['rowversion'] = array('name' => 'ROWVERSION',
                                        'type' => 'rowversion');

        $colspecs['hierarchyid'] = array('name' => 'HIERARCHYID',
                                         'type' => 'string');
        $colspecs['sql_variant'] = array('name' => 'SQL_VARIANT',
                                         'type' => 'text',
                                         'size' => 900);
        $colspecs['table']       = array('name' => 'TABLE',
                                         'type' => 'string');
        $colspecs['xml']         = array('name' => 'XML',
                                         'type' => 'string');
        $colspecs['uniqueidentifier'] = array('name' => 'UNIQUEIDENTIFIER',
                                              'type' => 'string',
                                              'size' => 36);

        // these are spatial types
        $colspecs['geometry']   = array('name' => 'GEOMETRY',
                                        'type' => 'geometry',
                                        'size' => 2147483647);
        $colspecs['geography']  = array('name' => 'GEOGRAPHY',
                                        'type' => 'geography',
                                        'size' => 2147483647);

        // these are here just for compatability with MySQL
        $colspecs['boolean']    = array('name' => 'BOOLEAN',
                                        'type' => 'boolean',
                                        'size' => 1);
        $colspecs['set']        = array('name' => 'SET',
                                        'type' => 'array');
        $colspecs['enum']       = array('name' => 'ENUM',
                                        'type' => 'array');
        $colspecs['mediumint']  = array('name' => 'MEDIUMINT',
                                        'type' => 'integer');
        $colspecs['tinytext']   = array('name' => 'TINYTEXT',
                                        'type' => 'string');
        $colspecs['mediumtext'] = array('name' => 'MEDIUMTEXT',
                                        'type' => 'string');
        $colspecs['longtext']   = array('name' => 'LONGTEXT',
                                        'type' => 'string');

        return $colspecs;

    } // ddl_getColumnSpecs

    // ****************************************************************************
    function ddl_showColumns ($dbname, $tablename)
    // obtain a list of column names within the selected database table.
    {
        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        $this->connect($dbname) or trigger_error('SQLSRV', E_USER_ERROR);

        $out_array = array();

        // connect to the selected database
        $this->query = "USE [$dbname]";
        $result = sqlsrv_query($this->dbconnect, $this->query) or trigger_error('SQLSRV', E_USER_ERROR);

        // build the query string and run it
        $this->query = "SELECT isc.*, sc.is_identity"
                     ." FROM sys.columns AS sc"
                     ." LEFT JOIN sys.tables AS st ON (st.object_id=sc.object_id)"
                     ." LEFT JOIN information_schema.columns AS isc ON (isc.table_catalog='$dbname' AND isc.table_name='$tablename' AND isc.column_name=sc.name)"
                     ." WHERE st.name='$tablename'"
                     ." ORDER BY ordinal_position";
        $result = sqlsrv_query($this->dbconnect, $this->query, array(), array('scrollable' => SQLSRV_CURSOR_STATIC)) or trigger_error('SQLSRV', E_USER_ERROR);

        $count = sqlsrv_num_rows($result);

        // write query to log file, if option is turned on
        logSqlQuery ($dbname, $tablename, $this->query, $count);

        $colspecs = $this->ddl_getColumnSpecs();

        // identify primary and other unique keys
        $tablekeys = $this->ddl_showTableKeys($dbname, $tablename);
        $pkey = array();  // primary key
        $ukey = array();  // candidate (unique) keys
        foreach ($tablekeys as $key => $spec) {
        	if (is_True($spec['is_primary'])) {
        	    $pkey[] = strtolower($spec['column_id']);
    	    } elseif (is_True($spec['is_unique'])) {
    	        $ukey[] = strtolower($spec['column_id']);
        	} // if
        } // foreach

        // convert result set into an associative array for each row
        while ($row = sqlsrv_fetch_array($result, SQLSRV_FETCH_ASSOC)) {
            $row = array_change_key_case($row, CASE_LOWER);
            // initialise all settings
            $columnarray = array();
            $columnarray['col_maxsize']         = NULL;
            $columnarray['col_unsigned']        = NULL;
            $columnarray['col_precision']       = NULL;
            $columnarray['col_scale']           = NULL;
            $columnarray['col_minvalue']        = NULL;
            $columnarray['col_maxvalue']        = NULL;
            $columnarray['col_auto_increment']  = NULL;
            $columnarray['col_key']             = NULL;

            $columnarray['column_id'] = $row['column_name'];
            $columnarray['col_type']  = $row['data_type'];
            if (in_array($columnarray['column_id'], $pkey)) {
            	$columnarray['col_key'] = 'PRI';
            } elseif (in_array($columnarray['column_id'], $ukey)) {
                $columnarray['col_key'] = 'UNI';
            } // if
            $columnarray['column_seq'] = $row['ordinal_position'];
            if (is_True($row['is_nullable'])) {
                $columnarray['col_null'] = 'Y';
            } else {
                $columnarray['col_null'] = 'N';
            } // if
            // look for default enclosed in "('" and "')"
            if (preg_match("/(?<=\(').+(?='\))/", $row['column_default'], $regs)) {
                $columnarray['col_default'] = $regs[0];
            } // if
            if (is_True($row['is_identity'])) {
                $columnarray['col_auto_increment'] = TRUE;
            } // if

            unset($precision, $scale, $minvalue, $maxvalue);
            $type  = $columnarray['col_type'];
    	    $specs = $colspecs[$type];

    	    if (isset($specs['size'])) {
                $columnarray['col_maxsize'] = $specs['size'];
            } // if

            if ($specs['type'] == 'integer') {
                if ($row['numeric_precision'] > 0) {
                    $precision                  = $row['numeric_precision'];
                    $scale                      = 0;  // no decimal places
                	$columnarray['col_maxsize'] = $row['numeric_precision'];
                } // if
            } // if

            if ($specs['type'] == 'string') {
                if (!is_null($row['character_maximum_length']) AND $row['character_maximum_length'] > 0) {
                	$columnarray['col_maxsize'] = $row['character_maximum_length'];
                } else {
            	    $columnarray['col_maxsize'] = $specs['size'];
                } // if
            } // if

            if ($specs['type'] == 'numeric') {
                $precision                    = $row['numeric_precision'];
                $columnarray['col_precision'] = $row['numeric_precision'];
                $columnarray['col_maxsize']   = $row['numeric_precision'] + 1;  // include sign
                $scale                        = $row['numeric_scale'];
                $columnarray['col_scale']     = $row['numeric_scale'];
                if ($row['numeric_scale'] > 0) {
                    $columnarray['col_maxsize'] = $columnarray['col_maxsize'] + 1;  // include decimal point
                } // if
            } // if

            // look for minimum value in $colspecs
            if (isset($specs['minvalue'])) {
                $minvalue = $specs['minvalue'];
            } else {
                if (isset($precision)) {
                    // minvalue includes negative sign
                    $minvalue = '-' . str_repeat('9', $precision);
                    if ($scale > 0) {
                        // adjust values to include decimal places
                        $minvalue = $minvalue / pow(10, $scale);
                    } // if
                } // if
            } // if
            if (isset($minvalue)) {
                $columnarray['col_minvalue'] = $minvalue;
            } // if

            // look for maximum value in $colspecs
            if (isset($specs['maxvalue'])) {
                $maxvalue = $specs['maxvalue'];
            } else {
                if (isset($precision)) {
                    // maxvalue has no positive sign
                    $maxvalue = str_repeat('9', $precision);
                    if ($scale > 0) {
                        // adjust values to include decimal places
                        $maxvalue = $maxvalue / pow(10, $scale);
                    } // if
                } // if
            } // if
            if (isset($maxvalue)) {
                $columnarray['col_maxvalue'] = (string)$maxvalue;
            } // if

            // some columns have the option of being used as BOOLEAN
//            if ($columnarray['col_maxsize'] == 1) {
//            	if ($columnarray['col_type'] == 'char') {
//                    $columnarray['col_type'] = 'char,boolean';
//                } // if
//            } elseif ($columnarray['col_type'] == 'smallint') {
//                $columnarray['col_type'] = 'smallint,boolean';
//            } // if

            $columnarray['col_type_native'] = $columnarray['col_type'];

            if ($columnarray['col_type'] == 'numeric' AND $scale == 0) {
            	$columnarray['col_type'] = 'integer';
            } // if

            $out_array[] = $columnarray;
        } // while

        sqlsrv_free_stmt($result);

        return $out_array;

    } // ddl_showColumns

    // ****************************************************************************
    function ddl_showDatabases ($dbprefix=null)
    // obtain a list of existing database names.
    {
        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        $this->connect() or trigger_error('SQLSRV', E_USER_ERROR);

        $array = array();

        // build the query string and run it
        $this->query = "SELECT * FROM sys.databases WHERE name NOT IN ('master','model','msdb','tempdb') ORDER BY name";
        $result = sqlsrv_query($this->dbconnect, $this->query) or trigger_error('SQLSRV', E_USER_ERROR);

        $count = sqlsrv_num_rows($result);

        // write query to log file, if option is turned on
        logSqlQuery (null, null, $this->query, $count);

        // convert result set into a simple indexed array for each row
        while ($row = sqlsrv_fetch_array($result, SQLSRV_FETCH_ASSOC)) {
            $array[] = $row['name'];
        } // while

        sqlsrv_free_stmt($result);

        return $array;

    } // ddl_showDatabases

    // ****************************************************************************
    function ddl_showTables ($dbname)
    // obtain a list of tables within the specified schema.
    {
        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        $this->connect($dbname) or trigger_error('SQLSRV', E_USER_ERROR);

        $array = array();

        // connect to the selected database
        $this->query = "USE [$dbname]";
        $result = sqlsrv_query($this->dbconnect, $this->query) or trigger_error('SQLSRV', E_USER_ERROR);

        // build the query string and run it
        $this->query = "SELECT * FROM INFORMATION_SCHEMA.tables WHERE table_catalog = '$dbname' ORDER BY table_name";


        $result = sqlsrv_query($this->dbconnect, $this->query, array(), array('scrollable' => SQLSRV_CURSOR_STATIC)) or trigger_error('SQLSRV', E_USER_ERROR);

        $count = sqlsrv_num_rows($result);

        // write query to log file, if option is turned on
        logSqlQuery ($dbname, null, $this->query, $count);

        // convert result set into an associative array for each row
        while ($row = sqlsrv_fetch_array($result, SQLSRV_FETCH_ASSOC)) {
            $array[] = $row['TABLE_NAME'];
        } // while

        sqlsrv_free_stmt($result);

        return $array;

    } // ddl_showTables

    // ****************************************************************************
    function ddl_showTableKeys ($dbname, $tablename)
    // obtain a list of keys (indexes) for this table.
    {
        $dmlobject = $this;  // ensure this is in $errcontext if an error occurs

        $this->connect($dbname) or trigger_error('SQLSRV', E_USER_ERROR);

        $array = array();

        // build the query string and run it
        $this->query = "SELECT tc.table_name, tc.constraint_type, tc.constraint_name, kcu.column_name, kcu.ordinal_position
                          FROM INFORMATION_SCHEMA.table_constraints AS tc
                     LEFT JOIN INFORMATION_SCHEMA.key_column_usage AS kcu ON (kcu.CONSTRAINT_NAME=tc.CONSTRAINT_NAME)
                         WHERE tc.table_catalog='$dbname' AND tc.table_name='$tablename'
                      ORDER BY tc.TABLE_NAME, tc.CONSTRAINT_TYPE, kcu.ORDINAL_POSITION";

        $result = sqlsrv_query($this->dbconnect, $this->query, array(), array('scrollable' => SQLSRV_CURSOR_STATIC)) or trigger_error('SQLSRV', E_USER_ERROR);

        $count = sqlsrv_num_rows($result);

        // write query to log file, if option is turned on
        logSqlQuery ($dbname, $tablename, $this->query, $count);

        // convert result set into a simple indexed array for each row
        while ($row = sqlsrv_fetch_array($result, SQLSRV_FETCH_ASSOC)) {
            $row = array_change_key_case($row, CASE_LOWER);
            if ($row['constraint_type'] == 'PRIMARY KEY') {
            	$row['key_name']   = 'PRIMARY';
            	$row['is_primary'] = TRUE;
            	$row['is_unique']  = TRUE;
            } else {
                $row['key_name']   = $row['constraint_name'];
                $row['is_primary'] = FALSE;
                $row['is_unique']  = TRUE;
            } // if
            $row['column_id']    = $row['column_name'];
            $row['seq_in_index'] = $row['ordinal_position'];
            $array[] = $row;
        } // while

        sqlsrv_free_stmt($result);

        return $array;

    } // ddl_showTableKeys

    // ****************************************************************************
    function _setDatabaseLock ($table_locks)
    // lock database tables identified in $string
    {
        foreach ($table_locks as $mode => $mode_array) {
            foreach ($mode_array as $table) {
                if (empty($string)) {
                    $string = "$table";
                } else {
                    $string .= ", $table";
                } // if
            } // foreach
        } // foreach

        // set locking level
        switch ($this->row_locks){
            case 'SH':
                switch (strtoupper($this->row_locks_supp)) {
                	case 'A':
                		$mode = 'ACCESS SHARE';
                		break;
                	case 'R':
                        $mode = 'ROW SHARE';
                	    break;
                	case 'UE':
                	    $mode = 'SHARE UPDATE EXCLUSIVE';
                	    break;
                	case 'RE':
                	    $mode = 'SHARE ROW EXCLUSIVE';
                	    break;
                	default:
                	    $mode = 'SHARE';
                		break;
                } // switch
                break;
            case 'EX':
                switch (strtoupper($this->row_locks_supp)) {
                	case 'A':
                		$mode = 'ACCESS EXCLUSIVE';
                		break;
                	case 'R':
                	    $mode = 'ROW EXCLUSIVE';
                	    break;
                    default:
                	    $mode = 'EXCLUSIVE';
                		break;
                } // switch
                break;
            default:
                $mode = 'SHARE';
        } // switch

        if (!empty($string)) {
//            $this->query = "LOCK TABLE $string IN $mode MODE";
//            $result = sqlsrv_query($this->dbconnect, $this->query) or trigger_error('SQLSRV', E_USER_ERROR);
//            // write query to log file, if option is turned on
//            logSqlQuery (null, null, $this->query);
//            $this->query = '';
//            return true;
        } // if

        return true;

    } // _setDatabaseLock

    // ****************************************************************************
    function __sleep ()
    // perform object clean-up before serialization
    {

        // get associative array of class variables
        $object_vars = get_object_vars($this);

        // remove unwanted variables
        //unset($object_vars['data_raw']);

        // convert to indexed array
        $object_vars = array_keys($object_vars);

        return $object_vars;

    } // __sleep

    // ****************************************************************************
    function __toString()
    // this is for use by the error handler
    {
        return $this->getErrorString();
    } // __toString

// ****************************************************************************
} // end class
// ****************************************************************************

?>
