<?php
// *****************************************************************************
// Copyright 2003-2005 by A J Marston <http://www.tonymarston.net>
// Copyright 2006-2023 by Radicore Software Limited <http://www.radicore.org>
// *****************************************************************************
// $Date: 2024-09-21 15:29:21 +0100 (Sat, 21 Sep 2024) $
// $Author: tony $
// $Revision: 1577 $
// *****************************************************************************

// This is my universal error handler.

// It can be called using code such as:
// ..do something.. or trigger_error("message", E_USER_ERROR);

// tell system what function to use as the error handler
set_error_handler('errorHandler');

$GLOBALS['xslt_error']  = null;

if (version_compare(phpversion(), '5.0.0', '<')) {
    // these do not exist in PHP 4, so define them manually
    define('E_STRICT', 2048);
    define('E_RECOVERABLE_ERROR', 4096);
    define('E_DEPRECATED', 8192);
    define('E_USER_DEPRECATED', 16384);
} else {
    // new constants were added in new versions of PHP 5
    if (version_compare(phpversion(), '5.2.0', '<')) {
        // this does not exist before 5.2, so define manually
        define('E_RECOVERABLE_ERROR', 4096);
    } // if
    if (version_compare(phpversion(), '5.3.0', '<')) {
        // these do not exist before 5.3, so define manually
        define('E_DEPRECATED', 8192);
        define('E_USER_DEPRECATED', 16384);
    } // if
} // if

// ****************************************************************************
//function errorHandler ($errno, $errstr, $errfile, $errline, $errcontext)
function errorHandler ($errno, $errstr, $errfile, $errline)
// If the error condition is E_USER_ERROR or above then abort
{
    if (version_compare(phpversion(), '8.0.0', '<')) {
        if (error_reporting() == 0) {
            return;  // call prepended with '@' symbol, so error reporting has been suppressed
        } // if
    } else {
        if (!(error_reporting() & $errno)) {
            return; // Silenced
        } // if
    } // if

    //if ($errno != E_STRICT) {
        if (function_exists('debugBreak')) {
            if (defined('DEBUGBREAK_ON_ERROR') AND DEBUGBREAK_ON_ERROR === true) {
                debugBreak();
            } // if
        } // if
    //} // if

    //logstuff("error: " .'(' .$errno .') ' .$errstr, $errfile, $errline);

    if ($errno == E_WARNING AND preg_match('/^(fread\(\)|fwrite\(\))/i', $errstr)) {
        if (preg_match('/(fail|abort)/i', $errstr)) {
            $errno = E_ERROR;  // convert from warning into a fatal error
        } // if
    } // if
    $errcontext = null;
    $trace = debug_backtrace(DEBUG_BACKTRACE_PROVIDE_OBJECT);
    if (version_compare(phpversion(), '8.0.0', '<')) {
        if (isset($trace[0]['args'][4])) {
            $errcontext =& $trace[0]['args'][4];
            if (!empty($errcontext['dmlobject'])) {
                $dmlobject = $errcontext['dmlobject'];  // allow data to be extracted from DML object
            } // if
        } // if
    } else {
        if (isset($trace[2]['object'])) {
            $dmlobject = $trace[2]['object'];  // allow data to be extracted from DML object
        } // if
        if (!empty($trace[1]) AND array_key_exists('function', $trace[1]) AND $trace[1]['function'] == 'require') {
            $errno = E_USER_ERROR;
        } // if
    } // if

    switch ($errno) {
        case E_USER_WARNING:
        case E_USER_NOTICE:
        case E_WARNING:
            if (isset($dmlobject) AND is_object($dmlobject)) {
                if ($dmlobject instanceof mysql) {
                    $errno  = $dmlobject->getErrorNo();
                    if ($errno <> 0) {
                        break;
                    } // if
                } elseif ($dmlobject instanceof pgsql) {
                    $errno  = $dmlobject->getErrorNo();
                    if ($errno <> 0) {
                        break;
                    } // if
                } elseif ($dmlobject instanceof oracle) {
                    $errno  = $dmlobject->getErrorNo();
                    if ($errno <> 0) {
                        if ($errno == 1002) {
                             // "fetch out of sequence" - IGNORE
                             $dmlobject->error_string = null;
                        } else {
                            break;
                        } // if
                    } // if
                } elseif ($dmlobject instanceof sqlsrv) {
                    $errno  = $dmlobject->getErrorNo();
                    if ($errno <> 0) {
                        break;
                    } // if
                } else {
                    // context is an object, but does it have a variable called $error_string?
                    if (method_exists($dmlobject, 'getErrorString')) {
                        // yes, but does it contain an error message?
                        $message = $dmlobject->getErrorString();
                        if (!empty($message)) {
                            $errstr = $message;
                            break;
                        } // if
                    } // if
                } // if
            } elseif (isset($GLOBALS['batch']) AND preg_match('/^(require)/i', $errstr)) {
                // detected a failed 'require()' in a batch job, so abort now
                break;
            } // if
        case E_NOTICE:
        case E_CORE_WARNING:
        case E_COMPILE_WARNING:
        case E_STRICT;
        case E_RECOVERABLE_ERROR;
        case E_DEPRECATED;
        case E_USER_DEPRECATED;
            // save error details before returning
            $GLOBALS['php_errorno']   = $errno;
            $GLOBALS['php_errormsg']  = $errstr;
            $GLOBALS['php_errorfile'] = $errfile;
            $GLOBALS['php_errorline'] = $errline;
            if (version_compare(phpversion(), '7.0.0', '>=')) {
                $errors = error_get_last();
                error_clear_last();
            } else {
                $handler = set_error_handler('var_dump', 0);
                @$undef_var;
                restore_error_handler($handler);
            } // if
            return;
        case E_USER_ERROR:
        case E_ERROR:
        case E_PARSE:
        case E_CORE_ERROR:
        case E_COMPILE_ERROR:
        default:
            break;
    } // switch

    if (isset($_SESSION['logon_user_id'])) {
        $logon_user_id = $_SESSION['logon_user_id'];
    } // if

    if (isset($dmlobject) AND is_object($dmlobject)) {
        // retrieve error details from DML object
        if (method_exists($dmlobject, 'getQuery')) {
            $query  = $dmlobject->getQuery();
        } // if
        if (method_exists($dmlobject, 'getErrorNo')) {
            $errno  = $dmlobject->getErrorNo();
        } // if
        if (method_exists($dmlobject, 'getErrorString')) {
            $errstr = $dmlobject->getErrorString();
        } // if
        if (method_exists($dmlobject, 'getErrorString2')) {
            $errstr2 = $dmlobject->getErrorString2();
        } // if
    } // if

    if (isset($GLOBALS['batch']) AND is_True($GLOBALS['batch']) OR preg_match('/cli/i', php_sapi_name())) {
    	// skip the next bit
    } else {
        if (function_exists('XMLRPC_errorhandler')) {
            // this is within XML-RPC or SOAP server, so skip next bit
        } else {
        	if (!isset($_SESSION)) {
                session_start();
            } // if
        } // if
    } // if

    if (!defined('RDC_WITHIN_ERROR_HANDLER')) {
        // flag which tells getLanguageText() not to error if entry not found
    	define('RDC_WITHIN_ERROR_HANDLER', true);
    } // if

    $errorstring = "<div class='error'>\n";
    $errorstring .= "<h2>" . date('Y-m-d H:i:s') . "</h2>\n";
    //$errorstring .= "<p>Fatal Error: $errstr (# $errno).</p>\n";
    $errorstring .= "<p>" .getLanguageText('sys0063', $errstr);
    if (!empty($errno)) {
    	$errorstring .= " (# $errno)";
    } // if
    $errorstring .= "</p>\n";
    if (!empty($query)) {
        //$errorstring .= "<p>SQL query: $query</p>\n";
        $errorstring .= "<p>" .getLanguageText('sys0064', $query) ."</p>\n";
    } // if
    //$errorstring .= "<p>Error in line $errline of file '$errfile'.</p>\n";
    $errorstring .= "<p>" .getLanguageText('sys0065', $errline, $errfile) ."</p>\n";
    if (!empty($errstr2)) {
    	$errorstring .= "<p>" .$errstr2 ."</p>\n";
    } // if
    //$errorstring .= "<p>PHP_SELF: '{$_SERVER['PHP_SELF']}'.</p>\n";
    $errorstring .= "<p>" .getLanguageText('sys0066', $_SERVER['PHP_SELF']) ."</p>\n";
    //$errorstring .= "<p>CURRENT DIRECTORY: " .getcwd() ."</p>\n";
    $errorstring .= "<p>" .getLanguageText('sys0187', getcwd()) ."</p>\n";

    $ip_address = getRealIPAddress();
    if (!empty($ip_address)) {
        //$errorstring .= "<p>SERVER_ADDR: '{$_SERVER['SERVER_ADDR']}'.</p>\n";
    	$errorstring .= "<p>" .getLanguageText('sys0183', $ip_address) ."</p>\n";
    } // if
    if (!empty($_SERVER['SERVER_NAME'])) {
    	//$errorstring .= "<p>SERVER_NAME: '{$_SERVER['SERVER_NAME']}'.</p>\n";
        $errorstring .= "<p>" .getLanguageText('sys0184', $_SERVER['SERVER_NAME']) ."</p>\n";
    } // if
    if (!empty($_SERVER['HTTP_HOST'])) {
    	//$errorstring .= "<p>HTTP_HOST: '{$_SERVER['HTTP_HOST']}'.</p>\n";
        $errorstring .= "<p>" .getLanguageText('sys0185', $_SERVER['HTTP_HOST']) ."</p>\n";
    } // if

    if (isset($logon_user_id)) {
        //$errorstring .= "<p>User Id: {$_SESSION['logon_user_id']}</p>\n";
        $errorstring .= "<p>" .getLanguageText('sys0067', $logon_user_id) ."</p>\n";
    } // if
    if (isset($_SESSION['role_id'])) {
        //$errorstring .= "<p>Role Id: {$_SESSION['role_id']}</p>\n";
        $errorstring .= "<p>" .getLanguageText('sys0109', $_SESSION['role_id']) ."</p>\n";
    } // if
    if (!empty($_SERVER['REMOTE_ADDR'])) {
    	//$errorstring .= "<p>REMOTE_ADDR: {$_SERVER['REMOTE_ADDR']}</p>\n";
        $errorstring .= "<p>" .getLanguageText('sys0068', $_SERVER['REMOTE_ADDR']) ."</p>\n";
    } // if
    if (!empty($_SERVER['REMOTE_HOST'])) {
    	//$errorstring .= "<p>REMOTE_HOST: {$_SERVER['REMOTE_HOST']}</p>\n";
        $errorstring .= "<p>" .getLanguageText('sys0186', $_SERVER['REMOTE_HOST']) ."</p>\n";
    } // if
    if (!empty($_SERVER['REQUEST_URI'])) {
    	//$errorstring .= "<p>REQUEST_URI: {$_SERVER['REQUEST_URI']}</p>\n";
        $errorstring .= "<p>" .getLanguageText('sys0069', $_SERVER['REQUEST_URI']) ."</p>\n";
    } // if
    $errorstring .= "</div>\n";

    global $page_stack;
    //$history = "<p>" .'Page Stack:' ."</p>\n";
    $history  = "<p>" .getLanguageText('sys0073') ."</p>\n";
    $ix = 0;
    if ($page_stack) {
        $history .= "<pre>\n";
        foreach ($page_stack as $script => $text) {
        	$history .= '  [' .$ix .'] ' .$script ."\n";
        	$ix++;
        } // foreach
        $history .= "</pre>\n";
    } // if

    // retrieve and format debug_backtrace for output
    if (!empty($GLOBALS['exception_trace']) AND is_array($GLOBALS['exception_trace'])) {
        $trace = $GLOBALS['exception_trace'];
        unset($GLOBALS['exception_trace']);
    } // if

    //$trace_hdg  = "<p>Backtrace:</p>\n";
    $trace_hdg  = "<p>" .getLanguageText('sys0075') ."</p>\n";
    $trace_out  = '';
    foreach ($trace as $level => $string) {
        $trace_out .= "$level: array =\n";
        if (is_array($string)) {
            $trace_out .= print_Trace($level, $string);
        } // if
    } // foreach

    // remove all html tags for email message
    $search = array('<br>', '<p>', '</p>', '<pre>', '</pre>');
    $mailstring  = strip_tags($errorstring);
    $mailstring .= str_replace($search, '', $history);
    $mailstring .= str_replace($search, '', $trace_hdg);
    $mailstring .= $trace_out;

    //$header = "<h1>This application has encountered an unrecoverable error</h1>\n";
    $header = "<h1>" .getLanguageText('sys0060') ."</h1>\n";
    //$header = "<p>The following has been reported to the administrator:</p>\n";
    $header .= "<p>"  .getLanguageText('sys0061') ."</p>\n";

    $errorstring = $header .$errorstring;

    if (function_exists('XMLRPC_errorhandler')) {
        // this is within an XML-RPC or SOAP server, so skip next bit
    } elseif (isset($GLOBALS['batch']) AND is_True($GLOBALS['batch'])) {
        // skip the next bit
    } else {
        if (preg_match('/^(127.0.0.1|localhost|desktop|laptop)$/i', $_SERVER['SERVER_NAME'])
        OR  preg_match('/^(local\.)/i', $_SERVER['SERVER_NAME'])
        OR  $_SERVER['REMOTE_ADDR'] == '62.30.82.227'
        OR (isset($GLOBALS['mode']) AND $GLOBALS['mode'] == 'batch')) {
?>
<!DOCTYPE HTML SYSTEM>
<HTML>
<HEAD>
<TITLE>RADICORE</TITLE>
<META http-equiv=Content-Type content="text/html; charset=utf-8">
<STYLE type="text/css">
<!--
  .error h2, .error p {
    color: red;
    font-weight: bold;
  }
-->
</STYLE>
</HEAD>
<BODY>
<?php echo $errorstring."\n"; ?>
</BODY>
</HTML>
<?php
        } else {
            //echo "<p>This application has encountered an unrecoverable error</p>";
            echo '<p>' .getLanguageText('sys0060') .'<br>';
            //echo "<p>Please contact your system administrator for details</p>";
            echo getLanguageText('sys0149') ."</p>\n";
        } // if
    } // if

    // remove line breaks
    //$errorstring = str_replace("\n", "", $errorstring);
    //$errorstring .= "\n"; // except the last one

    // add message to log file
    if (defined('ERROR_LOG')) {
    	$logfile = ERROR_LOG;  // write into specified file
    } elseif (defined('ERROR_LOG_DIR')) {
    	$logfile = ERROR_LOG_DIR.'/errorlog.html';  // write into specified directory
    } else {
        $logfile = $_SERVER['DOCUMENT_ROOT'].getParentDIR().'/error_logs/errorlog.html';  // write into default file
    } // if
    if (!empty($trace_out)) {
        $trace_out_x = htmlspecialchars($trace_out, ENT_COMPAT, 'UTF-8');
        if (empty($trace_out_x)) {
            $trace_out = preg_replace('/</', '&lt;', $trace_out);
        } // if
    } // if
    $output = "$errorstring\n$history\n$trace_hdg<pre>\n" .$trace_out ."\n</pre>";
    $res = error_log($output, 3, $logfile);

    global $stdout, $stdouth;  // handle for stdout
    if (isset($stdouth) AND is_resource($stdouth)) {
        if (ob_get_length() > 0) {
            $output .= "\n".ob_get_contents();
            ob_flush();                 // flush current output
        } // if
        send_to_stdout($output);
    } // if

    // send email to administrator
    if (defined('ERROR_LOG_MAILTO')) {
        $mailto = ERROR_LOG_MAILTO;
    } elseif (!empty($_SERVER['SERVER_ADMIN'])) {
        $mailto = $_SERVER['SERVER_ADMIN'];
    } else {
        $mailto = "webmaster@{$_SERVER['HTTP_HOST']}";
    } // if
    if (defined('ERROR_LOG_MAILFROM')) {
        $from = ERROR_LOG_MAILFROM;
    } else {
        $from = "webmaster@{$_SERVER['HTTP_HOST']}";
    } // if
    $subject = 'PHP error_log message';

    // Send the error to the administrator by email
    $res = send_email($from, $mailto, $subject, $mailstring);

    // rollback any incomplete transaction
    if (isset($dmlobject) AND is_object($dmlobject)) {
        if (method_exists($dmlobject, 'rollback')) {
            if (isset($dmlobject->schema)) {
                $dbname = $dmlobject->schema;
            } elseif (isset($dmlobject->dbname)) {
                $dbname = $dmlobject->dbname;
            } else {
                $dbname = null;
            } // if
            if (!empty($dbname)) {
                $dmlobject->rollback($dbname);
            } // if
        } // if
    } // if

    $GLOBALS['transaction_has_started'] = FALSE;

    if (isset($GLOBALS['batch']) AND is_True($GLOBALS['batch'])) {
    	// skip the next bit
    } else {
        if (isset($logon_user_id) AND !preg_match('/(internet|batch|webserver|RosettaNet)/i', $logon_user_id)) {
            // log the current user off
            if (!class_exists('logon')) {
            	require_once 'classes/logon.class.inc';
            } // if
            $dbobject = new logon;
            $errors = $dbobject->user_logoff($logon_user_id);
        } // if
    } // if

    if (isset($GLOBALS['batch']) AND is_True($GLOBALS['batch'])) {
        // continue
    } elseif (isset($logon_user_id) AND preg_match('/(internet|batch)/i', $logon_user_id)) {
        // continue
    } else {
        if (!empty($_SESSION)) {
        	@session_destroy();
            @session_unset();    // clear current session data
        } // if
    } // if

    if (function_exists('XMLRPC_errorhandler')) {
        // this is within XML-RPC or SOAP server, so send message back to client
    	XMLRPC_errorhandler();
    } // if

    die();

} // errorHandler

// *****************************************************************************
//function XML_errorHandler ($errno, $errstr, $errfile, $errline, $errcontext)
function XML_errorHandler ($errno, $errstr, $errfile, $errline, $errcontext=null)
// deal with errors from XML or XSL functions.
{
    if (empty($errcontext)) {
        $trace = debug_backtrace();
        $errcontext = $trace[0]['args'][4];
    } // if

    // pass these details to the standard error handler
    errorHandler (E_USER_ERROR, $errstr, $errfile, $errline, $errcontext);

} // XML_errorHandler

// ****************************************************************************
function XSLT_errorHandler ($parser, $errorno, $level, $fields)
// error handler for Sablotron XSLT process
{
    global $xslt_error;

    $xslt_error = "Error Number $errorno, Level $level, Fields:\n";
    if (is_array($fields)) {
        foreach ($fields as $key => $value) {
            $xslt_error .= "<br> $key => $value\n";
        } // while
    } else {
        $xslt_error .= "<br>$fields";
    } // if

    return;

} // XSLT_errorHandler

// ****************************************************************************
function exception_handler($exception)
// catch-all for unhandled exceptions.
{
    //$string = 'Uncaught exception from ' .get_class($exception) .', message = ' .$exception->getMessage();
    $string = getLanguageText('sys0062', get_class($exception), $exception->getMessage());

    $string .= ', code='.$exception->getCode();
    $string .= ', file='.$exception->getFile();
    $string .= ', line='.$exception->getLine();

    foreach(libxml_get_errors() as $error) {
        $string .= ', '.$error->message.', column '.$error->column;
    } // foreach

    if (method_exists($exception, 'getTrace')) {
        $trace1[] = array('exception' => $exception->getMessage(),
                          'file'      => $exception->getFile(),
                          'line'      => $exception->getLine(),
                          'code'      => $exception->getCode());
        $trace2 = $exception->getTrace();
//        foreach ($trace as $level) {
//            foreach ($level as $key => $value) {
//                if (is_string($value) OR is_long($value)) {
//                    $string .= ", $key: $value";
//                } elseif ($key == 'args' AND is_array($value)) {
//                    foreach ($value as $key1 => $value1) {
//                        if (is_object($value1)) {
//                            $value = ", object: ".get_class($value1);
//                        } // if
//                    } // foreach
//                    $string .= ', '.format_array($value);
//                } // if
//            } // foreach
//        } // foreach
        $trace = array_merge($trace1, $trace2);
        $GLOBALS['exception_trace'] = $trace;
    } // if

    trigger_error($string, E_USER_ERROR);

} // exception_handler

set_exception_handler('exception_handler');

// ****************************************************************************
function shutdown_handler()
// catch-all for errors not handled in standard errorHandler.
{
    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) AND $_SERVER['HTTP_X_REQUESTED_WITH'] == 'XMLHttpRequest') {
        $buffer = null;  // do not capture output from an AJAX request
    } else {
        $buffer = trim(ob_get_contents());
        if (preg_match('/^<!DOCTYPE /i', $buffer)) {
            $buffer = null;  // do not capture standard HTML output
        } // if
    } // if

    $error = error_get_last();
    if (!is_null($error)) {
        if ($error['type'] === E_WARNING OR $error['type'] === E_NOTICE) {
            $error = null;  // ignore these
        } // if
    } // if

    if (!empty($error) OR !empty($buffer)) {
        if (function_exists('debugBreak') AND defined("DEBUGBREAK_ON_ERROR")) {
            debugBreak();
        } // if
        // fatal error has occured
        if (defined('ERROR_LOG')) {
            $logfile = ERROR_LOG;  // write into specified file
        } elseif (defined('ERROR_LOG_DIR')) {
            $logfile = ERROR_LOG_DIR.'/errorlog.html';  // write into specified directory
        } else {
            $logfile = $_SERVER['DOCUMENT_ROOT'].getParentDIR().'/error_logs/errorlog.html';  // write into default file
        } // if
        $string = '['.date(DATE_RFC822).']'."<br>\n";
        $string .= 'script='.$_SERVER['SCRIPT_FILENAME']."<br>\n";
        if (is_array($error)) {
            foreach ($error as $key => $value) {
                $string .= "$key=$value <br>\n";
            } // foreach
        } // if
        if (!empty($buffer)) {
            $string .= "$buffer <br>\n";
        } // if
        $res = error_log("\n<p>$string</p>", 3, $logfile);
    } // if

} // shutdown_handler

register_shutdown_function('shutdown_handler');

?>
